// Face Registration JavaScript
// This file handles face detection and registration

let isModelLoaded = false;
let videoStream = null;
let captureInterval = null;

// Initialize face detection
async function initFaceDetection() {
    try {
        // Define model path
        const modelPath = base_url + 'backend/faceattendance/models';
        console.log('Attempting to load models from:', modelPath);
        
        // Test if model path is accessible
        try {
            const testResponse = await fetch(modelPath + '/test.txt');
            if (testResponse.ok) {
                console.log('Model path is accessible');
            } else {
                console.error('Model path is not accessible:', testResponse.status);
                return false;
            }
        } catch (testError) {
            console.error('Error testing model path:', testError);
            return false;
        }
        
        // Load required face-api.js models
        try {
            await Promise.all([
                faceapi.nets.tinyFaceDetector.loadFromUri(modelPath),
                faceapi.nets.faceLandmark68Net.loadFromUri(modelPath),
                faceapi.nets.faceRecognitionNet.loadFromUri(modelPath)
            ]);
            console.log('All models loaded successfully');
        } catch (error) {
            console.error('Error loading models:', error);
            return false;
        }
        
        isModelLoaded = true;
        console.log('Face detection models loaded successfully');
        return true;
    } catch (error) {
        console.error('Error initializing face detection:', error);
        return false;
    }
}

// Start camera for face capture
async function startCamera() {
    console.log('🎥 startCamera() function called - WORKING!');
    
    if (!isModelLoaded) {
        await initFaceDetection();
    }
    
    const video = document.getElementById('face-video');
    if (!video) {
        console.error('Video element not found');
        return false;
    }
    
    // Show camera container and loading indicator
    $('#face-capture-container').show();
    $('#camera-loading').show();
    
    // Start camera with settings
    try {
        // Get camera constraints from settings
        let constraints = { video: true };
        
        // Check if CameraSettings is available and get constraints
        if (typeof CameraSettings !== 'undefined') {
            console.log('CameraSettings class found, creating instance...');
            const cameraSettings = new CameraSettings();
            
            // Wait for settings to load completely
            await cameraSettings.waitForSettings();
            console.log('Settings loaded, getting constraints...');
            
            constraints = cameraSettings.getCameraConstraints();
            console.log('Using camera settings:', constraints);
        } else {
            // Fallback to default settings
            constraints = {
                video: {
                    width: 640,
                    height: 480,
                    facingMode: 'user' // Front camera
                }
            };
            console.log('Using default camera settings (CameraSettings not available)');
        }
        
        console.log('Final constraints being used:', JSON.stringify(constraints, null, 2));
        
        // Stop any existing stream first
        if (videoStream) {
            videoStream.getTracks().forEach(track => track.stop());
            videoStream = null;
        }
        
        const stream = await navigator.mediaDevices.getUserMedia(constraints);
        videoStream = stream;
        video.srcObject = stream;
        
        // Wait for video to be ready and play it
        await new Promise((resolve, reject) => {
            video.onloadedmetadata = () => {
                video.play().then(() => {
                    console.log('Camera started successfully');
                    resolve();
                }).catch((playError) => {
                    if (playError.name === 'AbortError') {
                        console.log('AbortError ignored - camera should still work');
                        resolve(); // Resolve anyway as camera usually works
                    } else {
                        reject(playError);
                    }
                });
            };
            
            video.onerror = (error) => {
                reject(new Error('Video loading failed'));
            };
        });
        $('#camera-loading').hide();
        
        // Show capture and stop buttons, hide start button
        $('#start-capture').hide();
        $('#capture-face').show();
        $('#stop-capture').show();
        
        // Create canvas for face detection display
        if (!document.getElementById('face-canvas')) {
            const canvas = faceapi.createCanvasFromMedia(video);
            canvas.id = 'face-canvas';
            canvas.style.position = 'absolute';
            canvas.style.top = '0';
            canvas.style.left = '0';
            canvas.style.zIndex = '2';
            document.getElementById('face-capture-container').appendChild(canvas);
        }
        
        // Start face detection
        startFaceDetection(video);
        
        return true;
    } catch (error) {
        console.error('Error starting camera:', error);
        $('#camera-loading').hide();
        
        // More user-friendly error messages
        let errorMessage = 'Error accessing camera: ';
        if (error.name === 'NotAllowedError') {
            errorMessage = 'Camera access denied. Please allow camera permissions and try again.';
        } else if (error.name === 'NotFoundError') {
            errorMessage = 'No camera found. Please check your camera connection.';
        } else if (error.name === 'NotReadableError') {
            errorMessage = 'Camera is being used by another application. Please close other apps and try again.';
        } else {
            errorMessage += error.message;
        }
        
        $('#status-message').text(errorMessage);
        return false;
    }
}

// Stop camera
function stopCamera() {
    if (captureInterval) {
        clearInterval(captureInterval);
        captureInterval = null;
    }
    
    if (videoStream) {
        videoStream.getTracks().forEach(track => track.stop());
        videoStream = null;
    }
    
    const video = document.getElementById('face-video');
    if (video) {
        video.srcObject = null;
    }
    
    // Remove canvas
    const canvas = document.getElementById('face-canvas');
    if (canvas) {
        canvas.remove();
    }
    
    // Hide camera container
    $('#face-capture-container').hide();
}

// Auto-capture variables
let autoCaptureTimer = null;
let faceDetectedCount = 0;
let isAutoCapturing = false;

// Start face detection - Mobile optimized with auto-capture
function startFaceDetection(video) {
    if (!isModelLoaded) {
        console.error('Face detection models not loaded');
        return;
    }
    
    const canvas = document.getElementById('face-canvas');
    const displaySize = { width: video.width, height: video.height };
    faceapi.matchDimensions(canvas, displaySize);
    
    // Mobile detection - reduce frequency for better performance
    const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    const detectionDelay = isMobile ? 300 : 100; // 300ms for mobile, 100ms for desktop
    
    detectionInterval = setInterval(async () => {
        try {
            const detections = await faceapi.detectAllFaces(video, new faceapi.TinyFaceDetectorOptions({
                inputSize: isMobile ? 320 : 416, // Smaller input size for mobile
                scoreThreshold: isMobile ? 0.6 : 0.5 // Higher threshold for mobile
            }))
                .withFaceLandmarks()
                .withFaceDescriptors();
            
            const resizedDetections = faceapi.resizeResults(detections, displaySize);
            canvas.getContext('2d').clearRect(0, 0, canvas.width, canvas.height);
            faceapi.draw.drawDetections(canvas, resizedDetections);
            faceapi.draw.drawFaceLandmarks(canvas, resizedDetections);
            
            // Auto-capture logic
            if (resizedDetections.length === 1 && !isAutoCapturing) {
                faceDetectedCount++;
                
                if (faceDetectedCount >= 3) { // Face detected for 3 consecutive frames
                    startAutoCapture();
                }
                
                $('#status-message').text(`Face detected! Auto-capture in ${Math.max(0, 3 - faceDetectedCount)} frames...`);
                $('#capture-face').prop('disabled', false);
            } else if (resizedDetections.length === 0) {
                faceDetectedCount = 0;
                cancelAutoCapture();
                $('#status-message').text('No face detected. Please position your face in the camera.');
                $('#capture-face').prop('disabled', true);
            } else if (resizedDetections.length > 1) {
                faceDetectedCount = 0;
                cancelAutoCapture();
                $('#status-message').text('Multiple faces detected. Please ensure only one face is visible.');
                $('#capture-face').prop('disabled', true);
            }
        } catch (error) {
            console.error('Error during face detection:', error);
        }
    }, detectionDelay);
}

// Start auto-capture countdown
function startAutoCapture() {
    if (isAutoCapturing) return;
    
    isAutoCapturing = true;
    let countdown = 3; // 3 second countdown
    
    // Get auto-capture delay from settings (default 3 seconds)
    let autoCaptureDelay = 3;
    if (typeof window.cameraSettings !== 'undefined' && window.cameraSettings.auto_capture_delay) {
        autoCaptureDelay = parseInt(window.cameraSettings.auto_capture_delay);
    }
    
    countdown = autoCaptureDelay;
    
    $('#status-message').text(`Auto-capture in ${countdown} seconds... Stay still!`);
    
    autoCaptureTimer = setInterval(() => {
        countdown--;
        if (countdown > 0) {
            $('#status-message').text(`Auto-capture in ${countdown} seconds... Stay still!`);
        } else {
            clearInterval(autoCaptureTimer);
            autoCaptureTimer = null;
            $('#status-message').text('Capturing face automatically...');
            
            // Automatically capture the face
            setTimeout(() => {
                captureFace();
                isAutoCapturing = false;
                faceDetectedCount = 0;
            }, 500);
        }
    }, 1000);
}

// Cancel auto-capture
function cancelAutoCapture() {
    if (autoCaptureTimer) {
        clearInterval(autoCaptureTimer);
        autoCaptureTimer = null;
    }
    isAutoCapturing = false;
}

// Capture face and extract descriptor
async function captureFace() {
    const video = document.getElementById('face-video');
    if (!video || !videoStream) {
        $('#status-message').text('Camera not active. Please start the camera first.');
        return;
    }
    
    try {
        // Get user ID and type
        const userType = $('#user-type').val();
        let userId;
        
        if (userType === 'student') {
            userId = $('#student-id').val();
            if (!userId) {
                $('#status-message').text('Please select a student.');
                return;
            }
        } else {
            userId = $('#staff-id').val();
            if (!userId) {
                $('#status-message').text('Please select a staff member.');
                return;
            }
        }
        
        // Update status
        $('#status-message').text('Processing face data...');
        
        // Detect face and get descriptor
        const detections = await faceapi.detectSingleFace(
            video, 
            new faceapi.TinyFaceDetectorOptions()
        ).withFaceLandmarks().withFaceDescriptor();
        
        if (!detections) {
            $('#status-message').text('No face detected. Please position your face properly.');
            return;
        }
        
        // Get face descriptor
        const descriptor = Array.from(detections.descriptor);
        
        // Save face data to database
        saveFaceDescriptor(userId, userType, descriptor);
    } catch (error) {
        console.error('Error capturing face:', error);
        $('#status-message').text('Error capturing face. Please try again.');
    }
}

// Process uploaded photo
async function processPhoto() {
    const fileInput = document.getElementById('photo-upload');
    if (!fileInput.files || fileInput.files.length === 0) {
        $('#status-message').text('Please select a photo first.');
        return;
    }
    
    try {
        // Get user ID and type
        const userType = $('#user-type').val();
        let userId;
        
        if (userType === 'student') {
            userId = $('#student-id').val();
            if (!userId) {
                $('#status-message').text('Please select a student.');
                return;
            }
        } else {
            userId = $('#staff-id').val();
            if (!userId) {
                $('#status-message').text('Please select a staff member.');
                return;
            }
        }
        
        // Show loading
        $('#photo-loading').show();
        $('#status-message').text('Processing uploaded photo...');
        
        const file = fileInput.files[0];
        
        // Create image element
        const img = new Image();
        img.onload = async function() {
            try {
                // Detect face and get descriptor from image
                const detections = await faceapi.detectSingleFace(
                    img, 
                    new faceapi.TinyFaceDetectorOptions({
                        inputSize: 416,
                        scoreThreshold: 0.5
                    })
                ).withFaceLandmarks().withFaceDescriptor();
                
                $('#photo-loading').hide();
                
                if (!detections) {
                    $('#status-message').text('No face detected in the uploaded photo. Please upload a clear photo with a visible face.');
                    return;
                }
                
                // Get face descriptor
                const descriptor = Array.from(detections.descriptor);
                
                // Save face data to database
                saveFaceDescriptor(userId, userType, descriptor);
            } catch (error) {
                $('#photo-loading').hide();
                console.error('Error processing photo:', error);
                $('#status-message').text('Error processing photo. Please try again.');
            }
        };
        
        img.onerror = function() {
            $('#photo-loading').hide();
            $('#status-message').text('Error loading image. Please select a valid image file.');
        };
        
        // Load image
        const reader = new FileReader();
        reader.onload = function(e) {
            img.src = e.target.result;
            // Also show preview
            $('#photo-preview').attr('src', e.target.result);
        };
        reader.readAsDataURL(file);
        
    } catch (error) {
        $('#photo-loading').hide();
        console.error('Error processing photo:', error);
        $('#status-message').text('Error processing photo. Please try again.');
    }
}

// Save face descriptor to database
function saveFaceDescriptor(userId, userType, descriptor) {
    $.ajax({
        url: base_url + 'admin/faceattendance/register/saveFaceData',
        type: 'POST',
        data: {
            userId: userId,
            userType: userType,
            faceDescriptor: JSON.stringify(descriptor)
        },
        dataType: 'json',
        success: function(response) {
            if (response.status) {
                $('#status-message').text('Face registered successfully!');
                
                // Reset form based on method
                const method = $('input[name="registration_method"]:checked').val();
                if (method === 'camera') {
                    // Stop camera after successful registration
                    stopCamera();
                    $('#start-capture').show();
                    $('#capture-face, #stop-capture').hide();
                } else {
                    // Clear photo upload
                    $('#photo-upload').val('');
                    $('#photo-preview').attr('src', '');
                    $('#process-photo-btn').hide();
                }
            } else {
                $('#status-message').text('Error: ' + response.msg);
            }
        },
        error: function() {
            $('#status-message').text('Error saving face data. Please try again.');
        }
    });
}

// Initialize when document is ready
$(document).ready(function() {
    // Initialize face detection
    initFaceDetection().then(success => {
        if (success) {
            $('#status-message').text('Face detection initialized. Select a user and click "Start Camera".');
        } else {
            $('#status-message').text('Failed to initialize face detection. Please check console for errors.');
            $('#start-capture').prop('disabled', true);
        }
    });
    
    // Apply select2 for better dropdown experience
    if ($.fn.select2) {
        $('#student-id, #staff-id, #class_filter, #section_filter').select2({
            width: '100%',
            placeholder: 'Select an option',
            allowClear: true
        });
    }
    
    // Toggle between student and staff selection
    $("#user-type").change(function() {
        if ($(this).val() === "student") {
            $("#student-selection").show();
            $("#staff-selection").hide();
        } else {
            $("#student-selection").hide();
            $("#staff-selection").show();
        }
    });
    
    // Toggle between camera and upload methods - Enhanced event handler
    $(document).on('change', "input[name='registration_method']", function() {
        console.log('Registration method changed to:', $(this).val()); // Debug log
        
        if ($(this).val() === "camera") {
            $("#camera-controls").show();
            $("#upload-controls").hide();
            $("#face-capture-container").show();
            $("#photo-preview-container").hide();
            $("#status-message").text("Select a user and click 'Start Camera' to begin face registration.");
        } else if ($(this).val() === "upload") {
            $("#camera-controls").hide();
            $("#upload-controls").show();
            $("#face-capture-container").hide();
            $("#photo-preview-container").show();
            $("#status-message").text("Select a user and upload a clear photo of their face.");
        }
    });
    
    // Also bind click event as backup
    $(document).on('click', "input[name='registration_method']", function() {
        $(this).trigger('change');
    });
    
    // Initialize default state and trigger change event
    setTimeout(function() {
        var defaultMethod = $("input[name='registration_method']:checked").val();
        console.log('Initial method detected:', defaultMethod);
        
        if (defaultMethod === "upload") {
            $("#camera-controls").hide();
            $("#upload-controls").show();
            $("#face-capture-container").hide();
            $("#photo-preview-container").show();
            $("#status-message").text("Select a user and upload a clear photo of their face.");
        }
        
        // Force trigger change event to ensure proper state
        $("input[name='registration_method']:checked").trigger('change');
    }, 500);
    
    // Handle class selection for section dropdown
    $("#class_filter").change(function() {
        var classId = $(this).val();
        getSections(classId);
    });
    
    // Function to get sections based on class selection
    function getSections(classId) {
        if (!classId) {
            // If no class selected, empty the sections dropdown
            $("#section_filter").html('<option value="">Select Section</option>');
            return;
        }
        
        // Show loading indicator
        $("#section_filter").html('<option value="">Loading...</option>');
        
        // Make AJAX request to get sections
        $.ajax({
            url: base_url + 'admin/faceattendance/register/getSections',
            type: 'POST',
            data: { class_id: classId },
            dataType: 'json',
            success: function(sections) {
                var options = '<option value="">Select Section</option>';
                
                if (sections && sections.length > 0) {
                    $.each(sections, function(index, section) {
                        options += '<option value="' + section.id + '">' + section.section + '</option>';
                    });
                }
                
                $("#section_filter").html(options);
            },
            error: function() {
                $("#section_filter").html('<option value="">Error loading sections</option>');
            }
        });
    }
    
    // Start camera button click
    $('#start-capture').click(function() {
        startCamera().then(success => {
            if (success) {
                $(this).hide();
                $('#capture-face, #stop-capture').show();
            }
        });
    });
    
    // Capture face button click
    $('#capture-face').click(function() {
        captureFace();
    });
    
    // Stop camera button click
    $('#stop-capture').click(function() {
        stopCamera();
        $(this).hide();
        $('#capture-face').hide();
        $('#start-capture').show();
        $('#status-message').text('Camera stopped. Click "Start Camera" to try again.');
    });
    
    // Photo upload change event
    $('#photo-upload').change(function() {
        const file = this.files[0];
        if (file) {
            // Update upload button text to show selected file
            $('#upload-button').html(`
                <i class="fa fa-check-circle" style="font-size: 24px; margin-bottom: 8px; display: block; color: #00ff00;"></i>
                <span style="font-size: 16px; display: block; margin-bottom: 5px;">Photo Selected: ${file.name}</span>
                <span style="font-size: 12px; opacity: 0.9;">Click to change photo</span>
            `);
            
            // Show process button - Fixed ID
            $('#process-photo-btn').show();
            
            // Show preview
            var reader = new FileReader();
            reader.onload = function(e) {
                $('#photo-preview').attr('src', e.target.result);
                $('#status-message').text('Photo selected. Click Process Photo to register the face.');
            };
            reader.readAsDataURL(file);
        } else {
            // Reset upload button to original state
            $('#upload-button').html(`
                <i class="fa fa-cloud-upload" style="font-size: 24px; margin-bottom: 8px; display: block;"></i>
                <span style="font-size: 16px; display: block; margin-bottom: 5px;">Click Here to Upload Photo</span>
                <span style="font-size: 12px; opacity: 0.9;">Choose a clear face image</span>
            `);
            
            $('#photo-preview').attr('src', '');
            $('#process-photo-btn').hide();
            $('#status-message').text('Select a user and upload a clear photo of their face.');
        }
    });
    
    // Process photo button click - Fixed ID selector
    $('#process-photo-btn').click(function() {
        console.log('Process Photo button clicked');
        processPhoto();
    });
    
    // Also bind to onclick attribute as backup
    window.processPhoto = processPhoto;
    
    // Make startCamera globally accessible
    window.startCamera = startCamera;
    
    // Add Start Camera button click handler
    $('#start-capture').click(function() {
        console.log('Button clicked via jQuery!');
        console.log('startCamera function exists:', typeof window.startCamera);
        try {
            window.startCamera();
        } catch(e) {
            console.error('Error calling startCamera:', e);
        }
    });
    
    // Add Capture Face button click handler
    $('#capture-face').click(function() {
        console.log('Capture Face button clicked');
        if (typeof captureFace === 'function') {
            captureFace();
        } else {
            console.error('captureFace function not found');
        }
    });
    
    // Add Stop Camera button click handler
    $('#stop-capture').click(function() {
        console.log('Stop Camera button clicked');
        if (typeof stopCamera === 'function') {
            stopCamera();
        } else {
            console.error('stopCamera function not found');
        }
    });
});
