// This file handles face detection and recognition for attendance

let isModelLoaded = false;
let faceMatcher = null;
let labeledFaceDescriptors = [];
let recognitionInterval = null;
let videoStream = null;

// Document ready event
$(document).ready(function() {
    console.log('Document ready');
    
    // Initialize face detection
    initFaceDetection().then(function(result) {
        if (result !== false) {
            isModelLoaded = true;
            console.log('Face detection initialized successfully');
        } else {
            console.error('Failed to initialize face detection');
        }
    });
    
    // Load face data
    loadFaceData().then(function(result) {
        if (result !== false) {
            console.log('Face data loaded successfully');
        } else {
            console.error('Failed to load face data');
        }
    });
    
    // Add event listeners for buttons
    $('#start-face-btn').on('click', function() {
        console.log('Start face recognition button clicked');
        startCamera();
        $(this).hide();
        $('#stop-face-btn').show();
    });
    
    // Add event listener for stop button
    $('#stop-face-btn').on('click', function() {
        console.log('Stop face recognition button clicked');
        stopFaceRecognition();
        $(this).hide();
        $('#start-face-btn').show();
    });
});

// Initialize face detection
async function initFaceDetection() {
    try {
        // Define model path without timestamp to ensure proper loading
        const modelPath = base_url + 'backend/faceattendance/models';
        console.log('Attempting to load models from:', modelPath);
        
        // Load models directly without testing path
        try {
            // Use hardcoded URLs to models to avoid path issues
            console.log('Loading TinyFaceDetector model...');
            await faceapi.nets.tinyFaceDetector.load(modelPath);
            console.log('TinyFaceDetector model loaded');
            
            console.log('Loading FaceLandmark68Net model...');
            await faceapi.nets.faceLandmark68Net.load(modelPath);
            console.log('FaceLandmark68Net model loaded');
            
            console.log('Loading FaceRecognitionNet model...');
            await faceapi.nets.faceRecognitionNet.load(modelPath);
            console.log('FaceRecognitionNet model loaded');
            
            console.log('All models loaded successfully');
        } catch (error) {
            console.error('Error loading models:', error);
            alert('Error loading face detection models: ' + error.message);
            return false;
        }
        
        isModelLoaded = true;
        console.log('Face detection models loaded successfully');
        
        // Load student/staff face data
        await loadFaceData();
        
        return true;
    } catch (error) {
        console.error('Error initializing face detection:', error);
        return false;
    }
}

// Load face data from server
async function loadFaceData() {
    try {
        console.log('Fetching face data from server...');
        const response = await fetch(base_url + 'admin/faceattendance/attendance/getFaceData');
        const data = await response.json();
        console.log('Server response:', data);
        
        if (data.status) {
            // Process face data
            labeledFaceDescriptors = [];
            console.log('Number of face records found:', data.faceData.length);
            
            // Only use registered faces, don't add test descriptor
            
            for (const person of data.faceData) {
                try {
                    console.log('Processing face data for person:', person.id, person.type);
                    if (person.faceDescriptor) {
                        console.log('Face descriptor found, attempting to parse...');
                        // Try to parse the face descriptor
                        let descriptorArray;
                        try {
                            descriptorArray = JSON.parse(person.faceDescriptor);
                            console.log('Successfully parsed descriptor as JSON');
                        } catch (parseError) {
                            console.log('JSON parse failed, trying alternative parsing method');
                            // If it's already a string representation of an array like '[0.1, 0.2, ...]'
                            // Remove the brackets and split by comma
                            const cleanString = person.faceDescriptor.replace('[', '').replace(']', '');
                            descriptorArray = cleanString.split(',').map(num => parseFloat(num.trim()));
                            console.log('Parsed descriptor using string splitting');
                        }
                        
                        // Ensure descriptor has correct length (128 values)
                        if (descriptorArray.length !== 128) {
                            console.warn('Descriptor has incorrect length:', descriptorArray.length);
                            // Pad or truncate to 128 values
                            if (descriptorArray.length < 128) {
                                // Pad with zeros
                                descriptorArray = [...descriptorArray, ...Array(128 - descriptorArray.length).fill(0)];
                            } else {
                                // Truncate
                                descriptorArray = descriptorArray.slice(0, 128);
                            }
                            console.log('Adjusted descriptor to length 128');
                        }
                        
                        console.log('Descriptor array length:', descriptorArray.length);
                        const descriptor = new Float32Array(descriptorArray);
                        const label = person.id + '_' + person.type; // student_id or staff_id with type
                        
                        labeledFaceDescriptors.push(
                            new faceapi.LabeledFaceDescriptors(label, [descriptor])
                        );
                        console.log('Successfully added face descriptor for:', label);
                    } else {
                        console.warn('Person has no face descriptor:', person.id, person.type);
                    }
                } catch (personError) {
                    console.error('Error processing person data:', personError, person);
                }
            }
            
            // Create face matcher with labeled face descriptors
            if (labeledFaceDescriptors.length > 0) {
                console.log('Creating face matcher with', labeledFaceDescriptors.length, 'descriptors');
                // Use a higher distance threshold for better matching
                faceMatcher = new faceapi.FaceMatcher(labeledFaceDescriptors, 0.8);
                console.log('Face data loaded successfully');
                return true;
            } else {
                console.warn('No valid face descriptors found in database');
                alert('No face data found in the database. Please register faces first.');
                return false;
            }
        } else {
            console.error('Failed to load face data:', data.msg);
            alert('Failed to load face data: ' + data.msg);
            return false;
        }
    } catch (error) {
        console.error('Error loading face data:', error);
        alert('Error loading face data: ' + error.message);
        return false;
    }
}

// Start camera
function startCamera() {
    console.log('Starting camera...');
    
    // Show alert to user
    alert('Starting camera. Please allow camera access when prompted.');
    
    const video = document.getElementById('face-video');
    if (!video) {
        console.error('Video element not found');
        alert('Video element not found. Please check the HTML structure.');
        return;
    }
    
    // Set video element styles to make sure it's visible
    video.style.width = '100%';
    video.style.height = '100%';
    video.style.objectFit = 'cover';
    video.style.display = 'block';
    
    // Request camera access with simpler constraints
    if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
        navigator.mediaDevices.getUserMedia({
            video: {
                width: { ideal: 640 },
                height: { ideal: 480 },
                facingMode: 'user'
            },
            audio: false
        })
        .then(function(stream) {
            console.log('Camera access granted');
            videoStream = stream;
            video.srcObject = stream;
            
            // Play video
            video.play();
            console.log('Video playing');
            
            // Start face recognition after a short delay
            setTimeout(() => {
                startFaceRecognition(video);
            }, 1000);
        })
        .catch(function(error) {
            console.error('Error accessing camera:', error);
            alert('Error accessing camera: ' + error.message + '\n\nPlease make sure you have granted camera permissions and no other application is using the camera.');
        });
    } else {
        console.error('getUserMedia not supported');
        alert('Your browser does not support camera access. Please try a different browser.');
    }
}

// Start face recognition
function startFaceRecognition(videoElement) {
    console.log('Starting face recognition...');
    if (!isModelLoaded) {
        console.error('Face models not loaded');
        alert('Face models not loaded. Please check the console for errors.');
        return;
    }
    
    // Create a default face matcher even if no faces are registered
    // This ensures the system can at least detect faces
    if (!faceMatcher) {
        console.warn('No face data available, creating default matcher');
        // Create a default descriptor for testing
        const defaultDescriptor = new Float32Array(128).fill(0.1);
        labeledFaceDescriptors = [
            new faceapi.LabeledFaceDescriptors('test_student', [defaultDescriptor])
        ];
        faceMatcher = new faceapi.FaceMatcher(labeledFaceDescriptors, 0.7); // Higher threshold for easier matching
    }
    
    // Clear any existing interval
    if (recognitionInterval) {
        clearInterval(recognitionInterval);
    }
    
    try {
        // Create canvas for face detection display
        const canvas = faceapi.createCanvasFromMedia(videoElement);
        document.getElementById('face-detection-container').appendChild(canvas);
        
        const displaySize = { width: videoElement.width, height: videoElement.height };
        faceapi.matchDimensions(canvas, displaySize);
        console.log('Canvas created with dimensions:', displaySize);
        
        // Add status display to the page
        const statusDiv = document.createElement('div');
        statusDiv.id = 'face-status';
        statusDiv.style.cssText = 'position: fixed; top: 10px; right: 10px; background-color: rgba(0,0,0,0.7); color: white; padding: 10px; border-radius: 5px; font-size: 16px; z-index: 9999;';
        statusDiv.innerHTML = 'Starting face detection...';
        document.body.appendChild(statusDiv);
        
        // Show initial alert
        alert('Face detection started. Please position your face in front of the camera.');
        
        // Counter for detection attempts
        let detectionAttempts = 0;
        let faceDetected = false;
        
        // Start recognition interval - slower interval (3 seconds) for better visibility
        recognitionInterval = setInterval(async () => {
            try {
                if (!videoElement.paused && !videoElement.ended) {
                    // Only increment attempts if face not yet detected
                    if (!faceDetected) {
                        detectionAttempts++;
                        console.log('Detecting faces... Attempt #' + detectionAttempts);
                        
                        // Update status display
                        statusDiv.innerHTML = 'Detecting faces... Attempt #' + detectionAttempts;
                        statusDiv.style.backgroundColor = 'rgba(0,0,150,0.7)';
                    }
                    
                    // Use simpler detection options for better visibility
                    const options = new faceapi.TinyFaceDetectorOptions({ 
                        inputSize: 320, // Larger input size for better detection
                        scoreThreshold: 0.3 // Lower threshold to detect more faces
                    });
                    
                    // Detect all faces first (simpler and faster)
                    const faceDetections = await faceapi.detectAllFaces(videoElement, options);
                    
                    // Clear canvas
                    const ctx = canvas.getContext('2d');
                    ctx.clearRect(0, 0, canvas.width, canvas.height);
                    
                    // Always show camera status
                    ctx.font = 'bold 16px Arial';
                    ctx.fillStyle = '#FFFFFF';
                    ctx.fillText('Camera Active - Attempt #' + detectionAttempts, 10, canvas.height - 10);
                    
                    if (faceDetections.length > 0) {
                        console.log('Detected', faceDetections.length, 'faces');
                        
                        // Update status display with success
                        statusDiv.innerHTML = 'FACE DETECTED! ✓';
                        statusDiv.style.backgroundColor = 'rgba(0,150,0,0.9)';
                        statusDiv.style.fontWeight = 'bold';
                        
                        // If this is the first face detection, show an alert and stop detection
                        if (!faceDetected) {
                            faceDetected = true;
                            alert('FACE DETECTED SUCCESSFULLY! Marking attendance...');
                            
                            // Stop the detection interval after a short delay
                            // This gives time for the attendance to be marked
                            setTimeout(() => {
                                if (recognitionInterval) {
                                    clearInterval(recognitionInterval);
                                    console.log('Face detection stopped after successful detection');
                                }
                            }, 2000);
                        }
                        
                        // Draw LARGE GREEN squares around faces instead of default detection
                        const resizedDetections = faceapi.resizeResults(faceDetections, displaySize);
                        
                        // Draw custom green boxes
                        resizedDetections.forEach(detection => {
                            const box = detection.box;
                            
                            // Draw thick green box
                            ctx.lineWidth = 6; // Even thicker
                            ctx.strokeStyle = '#00FF00'; // Bright green
                            ctx.strokeRect(box.x, box.y, box.width, box.height);
                            
                            // Add fill with transparency
                            ctx.fillStyle = 'rgba(0, 255, 0, 0.2)';
                            ctx.fillRect(box.x, box.y, box.width, box.height);
                            
                            // Add text inside the box
                            ctx.font = 'bold 16px Arial';
                            ctx.fillStyle = '#FFFFFF';
                            ctx.fillText('FACE DETECTED', box.x + 5, box.y + 20);
                        });
                        
                        // Show large success message on screen
                        ctx.font = 'bold 28px Arial';
                        ctx.fillStyle = '#00FF00';
                        ctx.fillText('FACE DETECTED SUCCESSFULLY!', 10, 40);
                        
                        // Add second line with attendance status
                        ctx.font = 'bold 24px Arial';
                        ctx.fillStyle = '#FFFFFF';
                        ctx.fillText('Marking attendance...', 10, 80);
                        
                        // Try to get full face data for the first face
                        try {
                            // Get full face data including landmarks and descriptor
                            const fullFaceDetection = await faceapi.detectSingleFace(videoElement, options)
                                .withFaceLandmarks()
                                .withFaceDescriptor();
                                
                            if (fullFaceDetection) {
                                console.log('Got face descriptor for detected face');
                                
                                // Draw success message
                                ctx.fillStyle = 'green';
                                ctx.fillText('Face Recognized!', 10, 60);
                                
                                // Try matching with registered faces
                                try {
                                    const descriptor = fullFaceDetection.descriptor;
                                    const match = faceMatcher.findBestMatch(descriptor);
                                    console.log('Face match result:', match.toString());
                                    
                                    // Only proceed if the face matches a registered face
                                    if (match.label !== 'unknown' && match.distance < 0.6) {
                                        // Extract ID and type from label
                                        // Handle different label formats
                                        let id, type;
                                        if (match.label.includes('_')) {
                                            [id, type] = match.label.split('_');
                                        } else {
                                            // Default to student type if not specified
                                            id = match.label;
                                            type = 'student';
                                        }
                                        
                                        console.log('Extracted ID:', id, 'Type:', type);
                                        
                                        // Get user name from the server
                                        try {
                                            const nameResponse = await fetch(base_url + 'admin/faceattendance/attendance/getUserName', {
                                                method: 'POST',
                                                headers: {
                                                    'Content-Type': 'application/x-www-form-urlencoded',
                                                },
                                                body: `id=${id}&type=${type}`
                                            });
                                            
                                            const nameData = await nameResponse.json();
                                            let userName = 'Unknown';
                                            
                                            if (nameData.status && nameData.name) {
                                                userName = nameData.name;
                                            }
                                            
                                            // Draw match result with user name
                                            const box = faceapi.resizeResults(fullFaceDetection.detection, displaySize).box;
                                            const drawBox = new faceapi.draw.DrawBox(box, { 
                                                label: userName,
                                                boxColor: 'green'
                                            });
                                            drawBox.draw(canvas);
                                            
                                            // Show large success message with user name
                                            ctx.font = 'bold 28px Arial';
                                            ctx.fillStyle = '#00FF00';
                                            ctx.fillText(`FACE DETECTED: ${userName}`, 10, 40);
                                            
                                            // Mark attendance
                                            console.log('Marking attendance for', type, 'with ID:', id);
                                            markAttendance(id, type);
                                        } catch (nameError) {
                                            console.error('Error getting user name:', nameError);
                                            
                                            // Draw match result without name
                                            const box = faceapi.resizeResults(fullFaceDetection.detection, displaySize).box;
                                            const drawBox = new faceapi.draw.DrawBox(box, { 
                                                label: `${type} ID: ${id}`,
                                                boxColor: 'green'
                                            });
                                            drawBox.draw(canvas);
                                            
                                            // Mark attendance
                                            console.log('Marking attendance for', type, 'with ID:', id);
                                            markAttendance(id, type);
                                        }
                                    } else {
                                        // Face not recognized or confidence too low
                                        console.log('Face not recognized or confidence too low');
                                        
                                        // Draw yellow box for unrecognized face
                                        const box = faceapi.resizeResults(fullFaceDetection.detection, displaySize).box;
                                        const drawBox = new faceapi.draw.DrawBox(box, { 
                                            label: 'Unregistered Face',
                                            boxColor: 'yellow'
                                        });
                                        drawBox.draw(canvas);
                                        
                                        // Show message that face is not registered
                                        ctx.font = 'bold 28px Arial';
                                        ctx.fillStyle = '#FFFF00';
                                        ctx.fillText('FACE NOT REGISTERED', 10, 40);
                                    }
                                } catch (matchError) {
                                    console.error('Error matching face:', matchError);
                                }
                            } else {
                                console.log('Could not get face descriptor');
                                // Show message that face detection failed
                                ctx.font = 'bold 28px Arial';
                                ctx.fillStyle = '#FF0000';
                                ctx.fillText('FACE DETECTION FAILED', 10, 40);
                                
                                // Show error in attendance status area
                                $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-warning')
                                    .html('<strong>Face detection failed!</strong><br>Please try again with better lighting and positioning.');
                            }
                        } catch (fullDetectionError) {
                            console.error('Error getting full face data:', fullDetectionError);
                            // Show error message
                            ctx.font = 'bold 28px Arial';
                            ctx.fillStyle = '#FF0000';
                            ctx.fillText('FACE DETECTION ERROR', 10, 40);
                            
                            // Show error in attendance status area
                            $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-warning')
                                .html('<strong>Face detection error!</strong><br>Please try again with better lighting and positioning.');
                        }
                    } else {
                        console.log('No faces detected in this frame');
                    }
                }
            } catch (detectionError) {
                console.error('Error during face detection:', detectionError);
            }
        }, 3000); // 3 second interval for better visibility
    } catch (error) {
        console.error('Error setting up face recognition:', error);
        alert('Error setting up face recognition. Please check the console for details.');
    }
}

// Stop face recognition
function stopFaceRecognition() {
    if (recognitionInterval) {
        clearInterval(recognitionInterval);
        recognitionInterval = null;
    }
    
    // Remove canvas
    const canvas = document.querySelector('#face-detection-container canvas');
    if (canvas) {
        canvas.remove();
    }
    
    // Remove status display
    const statusDiv = document.getElementById('face-status');
    if (statusDiv) {
        statusDiv.remove();
    }
    
    // Stop video stream
    if (videoStream) {
        videoStream.getTracks().forEach(track => track.stop());
        videoStream = null;
    }
}

// Mark attendance for recognized face
function markAttendance(id, type) {
    // Validate inputs
    if (!id || !type) {
        console.error('Invalid ID or type for attendance marking');
        $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-danger')
            .html('<strong>Error marking attendance!</strong><br>Invalid user data');
        return;
    }
    
    console.log('Marking attendance for', type, 'with ID:', id);

    // Clear any previous error messages
    $('#attendance-status').removeClass('alert-danger').addClass('alert-info')
        .html('<strong>Processing...</strong><br>Marking attendance, please wait...');

    // Make sure ID is a number
    const userId = parseInt(id, 10);
    if (isNaN(userId)) {
        console.error('Invalid ID format:', id);
        $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-danger')
            .html('<strong>Error marking attendance!</strong><br>Invalid user ID format');
        return;
    }
    
    // Ensure type is valid
    const userType = (type === 'student' || type === 'staff') ? type : 'student';
    
    // Log the final values being sent
    console.log('Sending attendance data:', { id: userId, type: userType });
    
    // Call the simpleMarkAttendance API
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/simpleMarkAttendance',
        type: 'POST',
        data: {
            id: userId,
            type: userType
        },
        dataType: 'json',
        success: function(response) {
            console.log('Attendance API response:', response);
            
            if (response.status) {
                // Check if attendance was already marked
                if (response.already_marked) {
                    console.log('Attendance was already marked for today');
                    // Get user name for thank you message
                    getUserNameAndShowThankYou(userId, userType, 'Attendance already marked for today');
                } else {
                    console.log('Attendance marked successfully');
                    // Get user name for thank you message
                    getUserNameAndShowThankYou(userId, userType, 'Attendance marked successfully');
                }
                
                // Clear any error messages
                $('#attendance-status').removeClass('alert-danger').addClass('alert-success')
                    .html('<strong>Success!</strong><br>' + (response.already_marked ? 'Attendance already marked for today' : 'Attendance marked successfully'));
            } else {
                console.error('Error marking attendance:', response.msg);
                // Show error in attendance status area
                $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-danger')
                    .html('<strong>Error marking attendance!</strong><br>' + response.msg);
            }
        },
        error: function(xhr, status, error) {
            console.error('AJAX error marking attendance:', error);
            // Show error in attendance status area
            $('#attendance-status').removeClass('alert-info alert-success').addClass('alert-danger')
                .html('<strong>Error marking attendance!</strong><br>Server error: ' + error);
        }
    });
}

// Function to get user name and show thank you message
function getUserNameAndShowThankYou(id, type, message) {
    // Get user name
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/getUserName',
        type: 'POST',
        data: {
            id: id,
            type: type
        },
        dataType: 'json',
        success: function(response) {
            if (response.status && response.name) {
                // Show thank you message with user name
                showThankYouMessage(response.name, message);
                
                // Also update the recent attendance list
                setTimeout(function() {
                    loadRecentAttendance();
                }, 1000);
            } else {
                // Show thank you message without name
                showThankYouMessage('User', message);
            }
        },
        error: function(xhr, status, error) {
            console.error('Error getting user name:', error);
            // Show thank you message without name
            showThankYouMessage('User', message);
        }
    });
}

// Function to show thank you message
function showThankYouMessage(name, message) {
    // Get the thank you container
    const thankYouContainer = document.getElementById('thank-you-container');
    
    if (!thankYouContainer) {
        console.error('Thank you container not found');
        return;
    }
    
    // Set the user name and message
    document.getElementById('thank-you-name').innerText = name;
    
    // Show the thank you message
    thankYouContainer.style.display = 'flex';
    
    // Add the success message
    const successMessage = document.getElementById('success-message');
    if (successMessage) {
        successMessage.innerText = 'Your attendance has been marked successfully!';
    }
    
    // Hide the thank you message after 5 seconds
    setTimeout(function() {
        thankYouContainer.style.display = 'none';
    }, 5000);
}

// Function to load recent attendance
function loadRecentAttendance() {
    // Get recent attendance from server
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/getRecentAttendance',
        type: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.status && response.data) {
                // Update the recent attendance list
                updateRecentAttendanceList(response.data);
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading recent attendance:', error);
        }
    });
}

// Function to update recent attendance list
function updateRecentAttendanceList(data) {
    const container = $('#recent-attendance-container');
    if (!container.length) {
        console.error('Recent attendance container not found');
        return;
    }
    
    // Clear existing items
    container.empty();
    
    // Add new items
    if (data.length === 0) {
        container.html('<div class="alert alert-info">No recent attendance records found.</div>');
        return;
    }
    
    // Create list
    const list = $('<div class="list-group"></div>');
    
    // Add items
    data.forEach(function(item) {
        const listItem = $(`
            <div class="list-group-item">
                <div class="row">
                    <div class="col-md-8">
                        <h4 class="list-group-item-heading">${item.name}</h4>
                        <p class="list-group-item-text">
                            <strong>ID:</strong> ${item.id}<br>
                            <strong>Date:</strong> ${item.date}<br>
                            <strong>Time:</strong> ${item.time}<br>
                            <strong>Method:</strong> ${item.method}<br>
                        </p>
                    </div>
                    <div class="col-md-4 text-right">
                        <span class="label label-success">Marked</span>
                    </div>
                </div>
            </div>
        `);
        
        list.append(listItem);
    });
    
    container.append(list);
}
