// This file handles face detection and recognition for checkout

// Get base URL from PHP or use default
const baseUrl = (typeof base_url !== 'undefined') ? base_url : '../../../';

let isModelLoaded = false;
let faceMatcher = null;
let labeledFaceDescriptors = [];
let recognitionInterval = null;
let videoStream = null;

// Document ready event
$(document).ready(function() {
    console.log('Document ready for checkout');
    
    // Initialize face detection
    initFaceDetection().then(function(result) {
        if (result !== false) {
            isModelLoaded = true;
            console.log('Face detection initialized successfully for checkout');
        } else {
            console.error('Failed to initialize face detection for checkout');
        }
    });
    
    // Load face data
    loadFaceData().then(function(result) {
        if (result !== false) {
            console.log('Face data loaded successfully for checkout');
        } else {
            console.error('Failed to load face data for checkout');
        }
    });
    
    // Camera size control buttons
    $('#camera-size-small').click(function() {
        setCameraSize('small');
    });
    
    $('#camera-size-medium').click(function() {
        setCameraSize('medium');
    });
    
    $('#camera-size-large').click(function() {
        setCameraSize('large');
    });
    
    $('#camera-size-fullscreen').click(function() {
        toggleFullscreen();
    });
    
    // Add event listeners for buttons
    $('#start-checkout-btn').on('click', function() {
        console.log('Start face checkout button clicked');
        startCamera();
        $(this).hide();
        $('#stop-checkout-btn').show();
    });
    
    // Add event listener for stop button
    $('#stop-checkout-btn').on('click', function() {
        console.log('Stop face checkout button clicked');
        stopFaceRecognition();
        $(this).hide();
        $('#start-checkout-btn').show();
    });
});

// Initialize face detection
async function initFaceDetection() {
    try {
        // Define model path without timestamp to ensure proper loading
        const modelPath = base_url + 'backend/faceattendance/models';
        console.log('Attempting to load models from:', modelPath);
        
        // Load models directly without testing path
        try {
            // Use hardcoded URLs to models to avoid path issues
            console.log('Loading TinyFaceDetector model...');
            await faceapi.nets.tinyFaceDetector.load(modelPath);
            console.log('TinyFaceDetector model loaded');
            
            console.log('Loading FaceLandmark68Net model...');
            await faceapi.nets.faceLandmark68Net.load(modelPath);
            console.log('FaceLandmark68Net model loaded');
            
            console.log('Loading FaceRecognitionNet model...');
            await faceapi.nets.faceRecognitionNet.load(modelPath);
            console.log('FaceRecognitionNet model loaded');
            
            console.log('All models loaded successfully');
        } catch (error) {
            console.error('Error loading models:', error);
            alert('Error loading face detection models: ' + error.message);
            return false;
        }
        
        isModelLoaded = true;
        console.log('Face detection models loaded successfully');
        
        // Load student/staff face data
        await loadFaceData();
        
        return true;
    } catch (error) {
        console.error('Error initializing face detection:', error);
        return false;
    }
}

// Load face data from server
async function loadFaceData() {
    try {
        console.log('Fetching face data from server...');
        const response = await fetch(base_url + 'admin/faceattendance/attendance/getFaceData');
        const data = await response.json();
        console.log('Server response:', data);
        
        if (data.status) {
            // Process face data
            labeledFaceDescriptors = [];
            console.log('Number of face records found:', data.faceData.length);
            
            for (const person of data.faceData) {
                try {
                    console.log('Processing face data for person:', person.id, person.type);
                    if (person.faceDescriptor) {
                        console.log('Face descriptor found, attempting to parse...');
                        // Try to parse the face descriptor
                        let descriptorArray;
                        try {
                            descriptorArray = JSON.parse(person.faceDescriptor);
                            console.log('Successfully parsed descriptor as JSON');
                        } catch (parseError) {
                            console.log('JSON parse failed, trying alternative parsing method');
                            // If it's already a string representation of an array like '[0.1, 0.2, ...]'
                            // Remove the brackets and split by comma
                            const cleanString = person.faceDescriptor.replace('[', '').replace(']', '');
                            descriptorArray = cleanString.split(',').map(num => parseFloat(num.trim()));
                            console.log('Parsed descriptor using string splitting');
                        }
                        
                        // Ensure descriptor has correct length (128 values)
                        if (descriptorArray.length !== 128) {
                            console.warn('Descriptor has incorrect length:', descriptorArray.length);
                            // Pad or truncate to 128 values
                            if (descriptorArray.length < 128) {
                                // Pad with zeros
                                descriptorArray = [...descriptorArray, ...Array(128 - descriptorArray.length).fill(0)];
                            } else {
                                // Truncate
                                descriptorArray = descriptorArray.slice(0, 128);
                            }
                            console.log('Adjusted descriptor to length 128');
                        }
                        
                        console.log('Descriptor array length:', descriptorArray.length);
                        const descriptor = new Float32Array(descriptorArray);
                        const label = person.id + '_' + person.type; // student_id or staff_id with type
                        
                        labeledFaceDescriptors.push(
                            new faceapi.LabeledFaceDescriptors(label, [descriptor])
                        );
                        console.log('Successfully added face descriptor for:', label);
                    } else {
                        console.warn('Person has no face descriptor:', person.id, person.type);
                    }
                } catch (personError) {
                    console.error('Error processing person data:', personError, person);
                }
            }
            
            // Create face matcher with labeled face descriptors
            if (labeledFaceDescriptors.length > 0) {
                console.log('Creating face matcher with', labeledFaceDescriptors.length, 'descriptors');
                // Use a higher distance threshold for better matching
                faceMatcher = new faceapi.FaceMatcher(labeledFaceDescriptors, 0.8);
                console.log('Face data loaded successfully');
                return true;
            } else {
                console.warn('No valid face descriptors found in database');
                alert('No face data found in the database. Please register faces first.');
                return false;
            }
        } else {
            console.error('Failed to load face data:', data.msg);
            alert('Failed to load face data: ' + data.msg);
            return false;
        }
    } catch (error) {
        console.error('Error loading face data:', error);
        alert('Error loading face data: ' + error.message);
        return false;
    }
}

// Start camera
function startCamera() {
    console.log('Starting camera for checkout...');
    
    const video = document.getElementById('face-video');
    if (!video) {
        console.error('Video element not found');
        alert('Video element not found. Please check the HTML structure.');
        return;
    }
    
    // Update status
    $('#camera-status').removeClass('alert-danger alert-success').addClass('alert-info')
        .html('<strong>Starting camera...</strong><br>Please allow camera access when prompted.');
    
    // Set video element styles to make sure it's visible
    video.style.width = '100%';
    video.style.height = '100%';
    video.style.objectFit = 'cover';
    video.style.display = 'block';
    
    // Request camera access with simpler constraints
    if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
        navigator.mediaDevices.getUserMedia({
            video: {
                width: { ideal: 640 },
                height: { ideal: 480 },
                facingMode: 'user'
            }
        })
        .then(function(stream) {
            videoStream = stream;
            video.srcObject = stream;
            
            // Wait for video to be ready
            video.onloadedmetadata = function() {
                // Set explicit dimensions on video element
                video.width = 640;
                video.height = 480;
                
                video.play()
                .then(function() {
                    console.log('Video playback started');
                    
                    // Wait a bit for the video to initialize properly
                    setTimeout(function() {
                        // Create canvas for detection visualization
                        const canvas = document.getElementById('face-canvas');
                        if (!canvas) {
                            const newCanvas = faceapi.createCanvasFromMedia(video);
                            newCanvas.id = 'face-canvas';
                            newCanvas.style.position = 'absolute';
                            newCanvas.style.top = '0';
                            newCanvas.style.left = '0';
                            document.getElementById('face-detection-container').appendChild(newCanvas);
                        }
                        
                        // Start face recognition
                        startFaceRecognition(video);
                    }, 500); // Wait 500ms for video to stabilize
                })
                .catch(function(error) {
                    console.error('Error playing video:', error);
                });
            };
        })
        .catch(function(error) {
            console.error('Error accessing camera:', error);
            alert('Error accessing camera: ' + error.message + '. Please check camera permissions.');
        });
    } else {
        console.error('getUserMedia not supported');
        alert('Your browser does not support camera access. Please use a modern browser.');
    }
}

// Start face recognition
function startFaceRecognition(videoElement) {
    console.log('Starting face recognition for checkout...');
    
    if (!isModelLoaded) {
        console.error('Face models not loaded');
        alert('Face detection models are not loaded. Please refresh the page and try again.');
        return;
    }
    
    if (!faceMatcher) {
        console.error('Face matcher not initialized');
        alert('Face matcher not initialized. Please make sure faces are registered in the system.');
        return;
    }
    
    // Clear any existing interval
    if (recognitionInterval) {
        clearInterval(recognitionInterval);
    }
    
    // Get canvas for drawing
    const canvas = document.getElementById('face-canvas');
    if (!canvas) {
        console.error('Canvas element not found');
        return;
    }
    
    // Set display size - use clientWidth/clientHeight instead of width/height
    const displaySize = { width: videoElement.clientWidth, height: videoElement.clientHeight };
    console.log('Display size:', displaySize);
    
    // Make sure we have valid dimensions
    if (displaySize.width === 0 || displaySize.height === 0) {
        // Use fixed dimensions if client dimensions are not available
        displaySize.width = 640;
        displaySize.height = 480;
        console.log('Using fixed display size:', displaySize);
    }
    
    // Match canvas dimensions to video
    faceapi.matchDimensions(canvas, displaySize);
    
    // Show status
    $('#camera-status').removeClass('alert-danger alert-success').addClass('alert-info')
        .html('<strong>Starting...</strong><br>Face detection is active. Please look at the camera.');
    
    // Start recognition interval
    recognitionInterval = setInterval(async () => {
        try {
            // Detect all faces with landmarks and descriptors
            const detections = await faceapi.detectAllFaces(
                videoElement, 
                new faceapi.TinyFaceDetectorOptions({ inputSize: 320, scoreThreshold: 0.5 })
            ).withFaceLandmarks().withFaceDescriptors();
            
            // Resize detections to match display size
            const resizedDetections = faceapi.resizeResults(detections, displaySize);
            
            // Clear canvas
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            // Draw detection results with better styling
            // Clear canvas first
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            // Do not draw face detections and landmarks
            // We're removing the green steps as requested
            // faceapi.draw.drawDetections(canvas, resizedDetections, { boxColor: 'lime', textColor: 'white', boxStyle: { lineWidth: 2 } });
            // faceapi.draw.drawFaceLandmarks(canvas, resizedDetections, { drawLines: true, color: 'aqua', lineWidth: 2 });
            
            // Update status message
            if (resizedDetections.length === 0) {
                $('#camera-status').removeClass('alert-danger alert-success').addClass('alert-info')
                    .html('<strong>Searching...</strong><br>No face detected. Please position your face in the camera.');
            } else if (resizedDetections.length > 1) {
                $('#camera-status').removeClass('alert-success alert-info').addClass('alert-danger')
                    .html('<strong>Too many faces!</strong><br>Multiple faces detected. Please ensure only one person is in front of the camera.');
            } else {
                // We have exactly one face, try to match it
                const detection = resizedDetections[0];
                const descriptor = detection.descriptor;
                
                // Match the descriptor against known faces
                const match = faceMatcher.findBestMatch(descriptor);
                console.log('Face match result:', match);
                
                if (match.label !== 'unknown') {
                    // We have a match, extract ID and type
                    const [id, type] = match.label.split('_');
                    
                    // Show matched user info with better styling
                    $('#camera-status').removeClass('alert-danger alert-info').addClass('alert-success')
                        .html(`<strong>Face recognized!</strong><br>ID: ${id}, Type: ${type}`);
                    
                    // Draw green box around face with name
                    const box = detection.detection.box;
                    const drawBox = new faceapi.draw.DrawBox(box, { 
                        label: `${type.toUpperCase()} ID: ${id}`,
                        boxColor: 'lime',
                        drawLabelOptions: { fontSize: 16, fontStyle: 'bold' }
                    });
                    drawBox.draw(canvas);
                    
                    // Draw success message on canvas
                    const ctx = canvas.getContext('2d');
                    ctx.font = 'bold 28px Arial';
                    ctx.fillStyle = '#00FF00';
                    ctx.fillText('FACE RECOGNIZED', 10, 40);
                    
                    // Mark checkout without stopping recognition
                    markCheckout(id, type);
                    
                    // Continue recognition without restarting
                    // Just update status for next person
                    setTimeout(() => {
                        $('#camera-status').removeClass('alert-danger alert-success').addClass('alert-info')
                            .html('<strong>Ready for next person</strong><br>Position the next person in front of the camera.');
                    }, 1000);
                } else {
                    // Unknown face - Draw red box with NOT MATCH label
                    const box = detection.detection.box;
                    const drawBox = new faceapi.draw.DrawBox(box, { 
                        label: 'NOT MATCH',
                        boxColor: 'red',
                        drawLabelOptions: { fontSize: 16, fontStyle: 'bold', padding: 10 }
                    });
                    drawBox.draw(canvas);
                    
                    // Draw NOT MATCH message on canvas
                    const ctx = canvas.getContext('2d');
                    ctx.font = 'bold 28px Arial';
                    ctx.fillStyle = '#FF0000';
                    ctx.fillText('FACE NOT REGISTERED', 10, 40);
                    
                    // Update status message
                    $('#camera-status').removeClass('alert-success alert-info').addClass('alert-danger')
                        .html('<strong>Unknown face!</strong><br>Your face is not recognized. Please register your face first.');
                    
                    // Show NOT MATCH popup (smaller size and shorter time)
                    const popup = $(`<div class="not-match-popup" style="position:fixed;top:50%;left:50%;transform:translate(-50%,-50%);background:rgba(255,0,0,0.9);color:white;padding:10px;border-radius:8px;text-align:center;z-index:9999;box-shadow:0 0 10px rgba(0,0,0,0.5);width:200px;">
                        <h3 style="margin:5px 0;">FACE NOT MATCH!</h3>
                        <p style="font-size:14px;margin:5px 0;">Not registered</p>
                        <div class="error-icon" style="font-size:30px;color:white;margin:5px 0;">✗</div>
                    </div>`);
                    
                    $('body').append(popup);
                    
                    // Remove popup after 1 second
                    setTimeout(function() {
                        popup.fadeOut(200, function() {
                            $(this).remove();
                        });
                    }, 1000);
                    
                    // IMPORTANT: Do NOT call markCheckout for unregistered faces
                }
            }
        } catch (error) {
            console.error('Error during face recognition:', error);
            $('#camera-status').removeClass('alert-success alert-info').addClass('alert-danger')
                .html('<strong>Error!</strong><br>An error occurred during face recognition: ' + error.message);
        }
    }, 100); // Run detection every 100ms for faster response
}

// Set camera size (small, medium, large)
function setCameraSize(size) {
    const container = $('#face-detection-container');
    const video = $('#face-video');
    const canvas = $('#face-canvas');
    
    // Reset all size buttons to default
    $('#camera-size-small, #camera-size-medium, #camera-size-large').removeClass('btn-primary').addClass('btn-default');
    
    // Set the appropriate size
    switch(size) {
        case 'small':
            container.css({
                'height': '200px',
                'transition': 'height 0.3s ease'
            });
            $('#camera-size-small').removeClass('btn-default').addClass('btn-primary');
            break;
            
        case 'medium':
            container.css({
                'height': '300px',
                'transition': 'height 0.3s ease'
            });
            $('#camera-size-medium').removeClass('btn-default').addClass('btn-primary');
            break;
            
        case 'large':
            container.css({
                'height': '450px',
                'transition': 'height 0.3s ease'
            });
            $('#camera-size-large').removeClass('btn-default').addClass('btn-primary');
            break;
    }
    
    // If we have an active video stream, update canvas dimensions
    if (videoStream) {
        // Wait for transition to complete
        setTimeout(() => {
            const canvas = document.getElementById('face-canvas');
            if (canvas) {
                // Match canvas to new container size
                canvas.width = container.width();
                canvas.height = container.height();
            }
        }, 300);
    }
}

// Toggle fullscreen mode
function toggleFullscreen() {
    const container = document.getElementById('face-detection-container');
    
    if (!document.fullscreenElement) {
        // Enter fullscreen
        if (container.requestFullscreen) {
            container.requestFullscreen();
        } else if (container.mozRequestFullScreen) { // Firefox
            container.mozRequestFullScreen();
        } else if (container.webkitRequestFullscreen) { // Chrome, Safari, Opera
            container.webkitRequestFullscreen();
        } else if (container.msRequestFullscreen) { // IE/Edge
            container.msRequestFullscreen();
        }
        
        // Add exit fullscreen button
        if (!$('#exit-fullscreen-btn').length) {
            const exitBtn = $('<button id="exit-fullscreen-btn" class="btn btn-warning" style="position:absolute;top:10px;right:10px;z-index:9999;"><i class="fa fa-compress"></i> Exit Fullscreen</button>');
            $(container).append(exitBtn);
            
            exitBtn.on('click', function() {
                if (document.exitFullscreen) {
                    document.exitFullscreen();
                } else if (document.mozCancelFullScreen) {
                    document.mozCancelFullScreen();
                } else if (document.webkitExitFullscreen) {
                    document.webkitExitFullscreen();
                } else if (document.msExitFullscreen) {
                    document.msExitFullscreen();
                }
            });
        }
    } else {
        // Exit fullscreen
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.mozCancelFullScreen) {
            document.mozCancelFullScreen();
        } else if (document.webkitExitFullscreen) {
            document.webkitExitFullscreen();
        } else if (document.msExitFullscreen) {
            document.msExitFullscreen();
        }
        
        // Remove exit button
        $('#exit-fullscreen-btn').remove();
    }
}

// Show thank you message
function showThankYouMessage(name, message) {
    // Create or update thank you message
    if ($('#thank-you-message').length === 0) {
        // Create new message
        const thankYouDiv = $('<div id="thank-you-message" class="alert alert-success" style="margin-top: 15px; text-align: center; font-size: 18px;">' +
            '<strong>Thank You, ' + name + '!</strong><br>' + message +
            '</div>');
        
        // Add to page before camera status
        $('#camera-status').before(thankYouDiv);
    } else {
        // Update existing message
        $('#thank-you-message').html('<strong>Thank You, ' + name + '!</strong><br>' + message);
    }
}

// Stop face recognition
function stopFaceRecognition() {
    console.log('Stopping face recognition...');
    
    // Clear recognition interval
    if (recognitionInterval) {
        clearInterval(recognitionInterval);
        recognitionInterval = null;
    }
    
    // Stop camera stream
    if (videoStream) {
        videoStream.getTracks().forEach(track => {
            track.stop();
        });
        videoStream = null;
    }
    
    // Clear video source
    const video = document.getElementById('face-video');
    if (video) {
        video.srcObject = null;
    }
    
    // Reset status
    $('#attendance-status').removeClass('alert-danger alert-success').addClass('alert-info')
        .html('<strong>Stopped</strong><br>Face recognition has been stopped.');
    
    // Show start button
    $('#stop-checkout-btn').hide();
    $('#start-checkout-btn').show();
}

// Mark checkout for recognized face
function markCheckout(id, type) {
    // Validate inputs
    if (!id || !type) {
        console.error('Invalid ID or type for checkout marking');
        $('#camera-status').removeClass('alert-info alert-success').addClass('alert-danger')
            .html('<strong>Error marking checkout!</strong><br>Invalid user data');
        return;
    }
    
    console.log('Marking checkout for', type, 'with ID:', id);

    // Clear any previous error messages
    $('#camera-status').removeClass('alert-danger').addClass('alert-info')
        .html('<strong>Processing...</strong><br>Marking checkout, please wait...');

    // Make sure ID is a number
    const userId = parseInt(id, 10);
    if (isNaN(userId)) {
        console.error('Invalid ID format:', id);
        $('#camera-status').removeClass('alert-info alert-success').addClass('alert-danger')
            .html('<strong>Error marking checkout!</strong><br>Invalid user ID format');
        return;
    }
    
    // Ensure type is valid
    const userType = (type === 'student' || type === 'staff') ? type : 'student';
    
    // Log the final values being sent
    console.log('Sending checkout data:', { id: userId, type: userType });
    
    // Call the markOutTime API
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/markOutTime',
        type: 'POST',
        data: {
            id: userId,
            type: userType
        },
        dataType: 'json',
        success: function(response) {
            console.log('Checkout API response:', response);
            
            if (response.status) {
                // Success - Draw large success message on canvas
                const canvas = document.getElementById('face-canvas');
                if (canvas) {
                    const ctx = canvas.getContext('2d');
                    // Clear previous text
                    ctx.clearRect(0, 0, canvas.width, 30); // Clear only top area for text
                    
                    // Draw success message
                    ctx.font = 'bold 28px Arial';
                    ctx.fillStyle = '#00FF00';
                    ctx.fillText('CHECKOUT SUCCESSFUL', 10, 40);
                    
                    // Play success sound
                    const audio = new Audio(base_url + 'backend/faceattendance/beep.js');
                    audio.play().catch(e => console.log('Audio play error:', e));
                }
                
                // Update status message
                $('#camera-status').removeClass('alert-danger alert-info').addClass('alert-success')
                    .html('<strong>Success!</strong><br>' + (response.already_checked_out ? 'Already checked out for today' : 'Checkout time recorded successfully'));
                
                // Show appropriate popup based on the response
                const userName = response.name || 'User';
                
                if (response.already_checked_out) {
                    // Get user image URL based on type
                    const userImageUrl = (type === 'student') ? 
                        `${baseUrl}uploads/student_images/${id}.jpg` : 
                        `${baseUrl}uploads/staff_images/${id}.jpg`;
                    
                    // Already checked out popup with lighter black background and green text - smaller size
                    const alreadyMarkedPopup = $(`<div class="already-marked-popup" style="position:fixed;bottom:20px;left:50%;transform:translateX(-50%);background:rgba(0,0,0,0.7);color:#4CAF50;padding:10px;border-radius:8px;text-align:center;z-index:9999;box-shadow:0 5px 15px rgba(0,0,0,0.5);width:175px;">
                        <h3 style="margin:3px 0;font-size:14px;color:#4CAF50;font-weight:bold;">Hello, ${type.charAt(0).toUpperCase() + type.slice(1)}</h3>
                        <h4 style="margin:3px 0;font-size:12px;color:#4CAF50;">${userName}</h4>
                        <p style="font-size:10px;margin:3px 0;color:#4CAF50;">Already checked out!</p>
                        <div style="width:25px;height:25px;background:#4CAF50;border-radius:50%;margin:5px auto;display:flex;align-items:center;justify-content:center;">
                            <span style="color:white;font-size:14px;line-height:1;">✓</span>
                        </div>
                    </div>`);
                    
                    $('body').append(alreadyMarkedPopup);
                    
                    // Remove popup after 1 second
                    setTimeout(function() {
                        alreadyMarkedPopup.remove();
                    }, 1000);
                } else {
                    // Get user image URL based on type
                    const userImageUrl = (type === 'student') ? 
                        `${baseUrl}uploads/student_images/${id}.jpg` : 
                        `${baseUrl}uploads/staff_images/${id}.jpg`;
                    
                    // Successful checkout popup with lighter black background and green text - smaller size
                    const successPopup = $(`<div class="thank-you-popup" style="position:fixed;bottom:20px;left:50%;transform:translateX(-50%);background:rgba(0,0,0,0.7);color:#4CAF50;padding:10px;border-radius:8px;text-align:center;z-index:9999;box-shadow:0 5px 15px rgba(0,0,0,0.5);width:175px;">
                        <h3 style="margin:3px 0;font-size:14px;color:#4CAF50;font-weight:bold;">Hello, ${type.charAt(0).toUpperCase() + type.slice(1)}</h3>
                        <h4 style="margin:3px 0;font-size:12px;color:#4CAF50;">${userName}</h4>
                        <p style="font-size:10px;margin:3px 0;color:#4CAF50;">Checkout successful!</p>
                        <div style="width:25px;height:25px;background:#4CAF50;border-radius:50%;margin:5px auto;display:flex;align-items:center;justify-content:center;">
                            <span style="color:white;font-size:14px;line-height:1;">✓</span>
                        </div>
                    </div>`);
                    
                    $('body').append(successPopup);
                    
                    // Remove popup after 1 second
                    setTimeout(function() {
                        successPopup.remove();
                    }, 1000);
                }
            } else {
                // Check if this is a user_not_found error
                if (response.user_not_found) {
                    // User not found - this is the same as an unregistered face
                    // Show NOT MATCH popup with lighter black background and green text - smaller size
                    const notMatchPopup = $(`<div class="not-match-popup" style="position:fixed;bottom:20px;left:50%;transform:translateX(-50%);background:rgba(0,0,0,0.7);color:#4CAF50;padding:10px;border-radius:8px;text-align:center;z-index:9999;box-shadow:0 5px 15px rgba(0,0,0,0.5);width:175px;">
                        <h3 style="margin:3px 0;font-size:14px;color:#4CAF50;font-weight:bold;">Face Not Recognized</h3>
                        <h4 style="margin:3px 0;font-size:12px;color:#4CAF50;">FACE NOT MATCH!</h4>
                        <p style="font-size:10px;margin:3px 0;color:#4CAF50;">No match found.</p>
                        <div style="width:25px;height:25px;background:#4CAF50;border-radius:50%;margin:5px auto;display:flex;align-items:center;justify-content:center;">
                            <span style="color:white;font-size:14px;line-height:1;">✗</span>
                        </div>
                    </div>`);
                    
                    $('body').append(notMatchPopup);
                    
                    // Remove popup after 1 second
                    setTimeout(function() {
                        notMatchPopup.remove();
                    }, 1000);
                } else {
                    // Other error
                    $('#camera-status').removeClass('alert-info alert-success').addClass('alert-danger')
                        .html('<strong>Error!</strong><br>' + (response.msg || 'Failed to mark checkout'));
                    
                    // Draw error message on canvas
                    const canvas = document.getElementById('face-canvas');
                    if (canvas) {
                        const ctx = canvas.getContext('2d');
                        ctx.clearRect(0, 0, canvas.width, 30); // Clear only top area for text
                        ctx.font = 'bold 28px Arial';
                        ctx.fillStyle = '#FF0000';
                        ctx.fillText('CHECKOUT FAILED', 10, 40);
                    }
                    
                    // Show NOT MATCH popup - smaller and positioned below the camera
                    const errorPopup = $(`<div class="error-popup" style="position:fixed;bottom:20px;left:50%;transform:translateX(-50%);background:rgba(0,0,0,0.9);color:white;padding:15px;border-radius:10px;text-align:center;z-index:9999;box-shadow:0 5px 15px rgba(0,0,0,0.5);width:350px;">
                        <h3 style="margin:5px 0;font-size:20px;color:#FF5555;font-weight:bold;">Face Not Recognized</h3>
                        <h4 style="margin:5px 0;font-size:18px;color:white;">FACE NOT MATCH!</h4>
                        <p style="font-size:14px;margin:5px 0;color:white;">Your face could not be matched with any registered user.</p>
                        <div style="width:50px;height:50px;background:#FF5555;border-radius:50%;margin:10px auto;display:flex;align-items:center;justify-content:center;">
                            <span style="color:white;font-size:25px;line-height:1;">✗</span>
                        </div>
                    </div>`);
                    
                    $('body').append(errorPopup);
                    
                    // Remove popup after 1 second
                    setTimeout(function() {
                        errorPopup.fadeOut(200, function() {
                            $(this).remove();
                        });
                    }, 1000);
                }
            }
        },
        error: function(xhr, status, error) {
            console.error('AJAX error marking checkout:', error);
            $('#camera-status').removeClass('alert-info alert-success').addClass('alert-danger')
                .html('<strong>Error!</strong><br>Server error: ' + error);
            
            // Show NOT MATCH popup with lighter black background and green text - smaller size
            const errorPopup = $(`<div class="error-popup" style="position:fixed;bottom:20px;left:50%;transform:translateX(-50%);background:rgba(0,0,0,0.7);color:#4CAF50;padding:10px;border-radius:8px;text-align:center;z-index:9999;box-shadow:0 5px 15px rgba(0,0,0,0.5);width:175px;">
                <h3 style="margin:3px 0;font-size:14px;color:#4CAF50;font-weight:bold;">Face Not Recognized</h3>
                <h4 style="margin:3px 0;font-size:12px;color:#4CAF50;">FACE NOT MATCH!</h4>
                <p style="font-size:10px;margin:3px 0;color:#4CAF50;">No match found.</p>
                <div style="width:25px;height:25px;background:#4CAF50;border-radius:50%;margin:5px auto;display:flex;align-items:center;justify-content:center;">
                    <span style="color:white;font-size:14px;line-height:1;">✗</span>
                </div>
            </div>`);
            
            $('body').append(errorPopup);
            
            // Remove popup after 1 second
            setTimeout(function() {
                errorPopup.remove();
            }, 1000);
        }
    });
}

// Function to get user name and show thank you message
function getUserNameAndShowThankYou(id, type, message) {
    // Get user name
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/getUserName',
        type: 'POST',
        data: {
            id: id,
            type: type
        },
        dataType: 'json',
        success: function(response) {
            console.log('Get user name response:', response);
            
            if (response.status) {
                // Show thank you message with user name
                showThankYouMessage(response.name, message);
                
                // Update recent attendance list
                loadRecentAttendance();
            } else {
                console.error('Error getting user name:', response.msg);
                // Show generic thank you message
                showThankYouMessage('User', message);
            }
        },
        error: function(xhr, status, error) {
            console.error('AJAX error getting user name:', error);
            // Show generic thank you message
            showThankYouMessage('User', message);
        }
    });
}

// Function to show thank you message
function showThankYouMessage(name, message) {
    // Play thank you sound
    const audio = new Audio(base_url + 'backend/faceattendance/thank-you.mp3');
    audio.play();
    
    // Show thank you message
    $('#thank-you-name').text(name);
    $('#thank-you-message').text(message);
    $('#thank-you-time').text(new Date().toLocaleTimeString());
    $('#thank-you-modal').modal('show');
    
    // Hide modal after 5 seconds
    setTimeout(function() {
        $('#thank-you-modal').modal('hide');
        
        // Reset face recognition
        setTimeout(function() {
            // Restart face recognition if still on the page
            if ($('#start-checkout-btn').is(':hidden')) {
                startFaceRecognition(document.getElementById('face-video'));
            }
        }, 500);
    }, 5000);
}

// Function to load recent attendance
function loadRecentAttendance() {
    $.ajax({
        url: base_url + 'admin/faceattendance/attendance/getRecentAttendance',
        type: 'GET',
        dataType: 'json',
        success: function(response) {
            console.log('Recent attendance response:', response);
            
            if (response.status) {
                // Update recent attendance list
                updateRecentAttendanceList(response.data);
            } else {
                console.error('Error loading recent attendance:', response.msg);
            }
        },
        error: function(xhr, status, error) {
            console.error('AJAX error loading recent attendance:', error);
        }
    });
}

// Function to update recent attendance list
function updateRecentAttendanceList(data) {
    const $list = $('#recent-attendance-list');
    $list.empty();
    
    if (data.length === 0) {
        $list.append('<li class="list-group-item">No recent attendance records</li>');
        return;
    }
    
    // Add each attendance record to the list
    data.forEach(function(item) {
        const time = new Date(item.created_at).toLocaleTimeString();
        const date = new Date(item.created_at).toLocaleDateString();
        
        let statusBadge = '';
        if (item.has_checked_out) {
            statusBadge = '<span class="badge badge-success">Checked Out</span>';
        } else {
            statusBadge = '<span class="badge badge-primary">Checked In</span>';
        }
        
        const $item = $(`
            <li class="list-group-item">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong>${item.name}</strong> (${item.type})
                        <br>
                        <small>${date} ${time}</small>
                    </div>
                    <div>
                        ${statusBadge}
                    </div>
                </div>
            </li>
        `);
        
        $list.append($item);
    });
}
