<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Certificate_log_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->current_session = $this->setting_model->getCurrentSession();
        
        // Create certificate_logs table if it doesn't exist
        $this->createCertificateLogsTable();
    }
    
    /**
     * Create certificate_logs table if it doesn't exist
     */
    private function createCertificateLogsTable()
    {
        if (!$this->db->table_exists('certificate_logs')) {
            $this->load->dbforge();
            
            $fields = array(
                'id' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => TRUE,
                    'auto_increment' => TRUE
                ),
                'certificate_id' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => FALSE,
                ),
                'student_id' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => FALSE,
                ),
                'class_id' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => TRUE,
                ),
                'section_id' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => TRUE,
                ),
                'admission_no' => array(
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                    'null' => TRUE,
                ),
                'student_name' => array(
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => TRUE,
                ),
                'certificate_name' => array(
                    'type' => 'VARCHAR',
                    'constraint' => 255,
                    'null' => TRUE,
                ),
                'generated_by' => array(
                    'type' => 'INT',
                    'constraint' => 11,
                    'null' => TRUE,
                ),
                'created_at' => array(
                    'type' => 'DATETIME',
                    'null' => TRUE,
                ),
            );
            
            $this->dbforge->add_field($fields);
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->create_table('certificate_logs');
        }
    }
    
    /**
     * Add certificate log entry
     * 
     * @param array $data Log data
     * @return int Inserted ID
     */
    public function add($data)
    {
        $this->db->trans_start();
        $this->db->insert('certificate_logs', $data);
        $insert_id = $this->db->insert_id();
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            $this->db->trans_rollback();
            return FALSE;
        } else {
            return $insert_id;
        }
    }
    
    /**
     * Get all certificate logs
     * 
     * @param int $limit Limit results
     * @param int $offset Offset for pagination
     * @return array Certificate logs
     */
    public function getAll($limit = 100, $offset = 0)
    {
        // Simplified query to just get the basic log data without joins
        $this->db->select('*');
        $this->db->from('certificate_logs');
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result_array();
    }
    
    /**
     * Get certificate logs count
     * 
     * @return int Count
     */
    public function getCount()
    {
        return $this->db->count_all('certificate_logs');
    }
    
    /**
     * Search certificate logs
     * 
     * @param array $params Search parameters
     * @param int $limit Limit results
     * @param int $offset Offset for pagination
     * @return array Certificate logs
     */
    public function search($params = array(), $limit = 100, $offset = 0)
    {
        $this->db->select('cl.*, c.certificate_name as certificate_template_name, 
                          CONCAT(s.firstname, " ", IFNULL(s.middlename, ""), " ", IFNULL(s.lastname, "")) as full_student_name,
                          cl.student_name, cl.admission_no, cl.created_at,
                          CONCAT(staff.name, " ", IFNULL(staff.surname, "")) as staff_name,
                          classes.class, sections.section');
        $this->db->from('certificate_logs as cl');
        $this->db->join('certificates as c', 'c.id = cl.certificate_id', 'left');
        $this->db->join('students as s', 's.id = cl.student_id', 'left');
        $this->db->join('staff', 'staff.id = cl.generated_by', 'left');
        $this->db->join('classes', 'classes.id = cl.class_id', 'left');
        $this->db->join('sections', 'sections.id = cl.section_id', 'left');
        
        // Apply search filters
        if (!empty($params['certificate_id'])) {
            $this->db->where('cl.certificate_id', $params['certificate_id']);
        }
        
        if (!empty($params['class_id'])) {
            $this->db->where('cl.class_id', $params['class_id']);
        }
        
        if (!empty($params['section_id'])) {
            $this->db->where('cl.section_id', $params['section_id']);
        }
        
        if (!empty($params['student_id'])) {
            $this->db->where('cl.student_id', $params['student_id']);
        }
        
        if (!empty($params['admission_no'])) {
            $this->db->like('cl.admission_no', $params['admission_no']);
        }
        
        if (!empty($params['student_name'])) {
            $this->db->group_start();
            $this->db->like('cl.student_name', $params['student_name']);
            $this->db->or_like('CONCAT(s.firstname, " ", IFNULL(s.middlename, ""), " ", IFNULL(s.lastname, ""))', $params['student_name']);
            $this->db->group_end();
        }
        
        if (!empty($params['date_from']) && !empty($params['date_to'])) {
            $this->db->where('DATE(cl.created_at) >=', date('Y-m-d', strtotime($params['date_from'])));
            $this->db->where('DATE(cl.created_at) <=', date('Y-m-d', strtotime($params['date_to'])));
        } else if (!empty($params['date_from'])) {
            $this->db->where('DATE(cl.created_at) >=', date('Y-m-d', strtotime($params['date_from'])));
        } else if (!empty($params['date_to'])) {
            $this->db->where('DATE(cl.created_at) <=', date('Y-m-d', strtotime($params['date_to'])));
        }
        
        $this->db->order_by('cl.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result_array();
    }
    
    /**
     * Get search count
     * 
     * @param array $params Search parameters
     * @return int Count
     */
    public function getSearchCount($params = array())
    {
        $this->db->from('certificate_logs as cl');
        
        // Apply search filters
        if (!empty($params['certificate_id'])) {
            $this->db->where('cl.certificate_id', $params['certificate_id']);
        }
        
        if (!empty($params['class_id'])) {
            $this->db->where('cl.class_id', $params['class_id']);
        }
        
        if (!empty($params['section_id'])) {
            $this->db->where('cl.section_id', $params['section_id']);
        }
        
        if (!empty($params['student_id'])) {
            $this->db->where('cl.student_id', $params['student_id']);
        }
        
        if (!empty($params['admission_no'])) {
            $this->db->like('cl.admission_no', $params['admission_no']);
        }
        
        if (!empty($params['student_name'])) {
            $this->db->join('students as s', 's.id = cl.student_id', 'left');
            $this->db->group_start();
            $this->db->like('cl.student_name', $params['student_name']);
            $this->db->or_like('CONCAT(s.firstname, " ", IFNULL(s.middlename, ""), " ", IFNULL(s.lastname, ""))', $params['student_name']);
            $this->db->group_end();
        }
        
        if (!empty($params['date_from']) && !empty($params['date_to'])) {
            $this->db->where('DATE(cl.created_at) >=', date('Y-m-d', strtotime($params['date_from'])));
            $this->db->where('DATE(cl.created_at) <=', date('Y-m-d', strtotime($params['date_to'])));
        } else if (!empty($params['date_from'])) {
            $this->db->where('DATE(cl.created_at) >=', date('Y-m-d', strtotime($params['date_from'])));
        } else if (!empty($params['date_to'])) {
            $this->db->where('DATE(cl.created_at) <=', date('Y-m-d', strtotime($params['date_to'])));
        }
        
        return $this->db->count_all_results();
    }
}
