<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Accounts_model extends MY_Model
{

    public function __construct()
    {
        parent::__construct();
        $this->current_session = $this->setting_model->getCurrentSession();
    }

    /**
     * Get all accounts
     */
    public function getAllAccounts()
    {
        $this->db->select('*');
        $this->db->from('accounts');
        $this->db->order_by('account_number', 'ASC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get single account
     */
    public function getAccount($id)
    {
        $this->db->select('*');
        $this->db->from('accounts');
        $this->db->where('id', $id);
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Add new account
     */
    public function addAccount($data)
    {
        $this->db->trans_start();
        
        $this->db->insert('accounts', $data);
        $insert_id = $this->db->insert_id();
        
        $message = INSERT_RECORD_CONSTANT . " On accounts id " . $insert_id;
        $action = "Insert";
        $this->log($message, $insert_id, $action);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === false) {
            $this->db->trans_rollback();
            return false;
        } else {
            return $insert_id;
        }
    }

    /**
     * Update account
     */
    public function updateAccount($data)
    {
        $this->db->trans_start();
        
        $this->db->where('id', $data['id']);
        $this->db->update('accounts', $data);
        
        $message = UPDATE_RECORD_CONSTANT . " On accounts id " . $data['id'];
        $action = "Update";
        $this->log($message, $data['id'], $action);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === false) {
            $this->db->trans_rollback();
            return false;
        } else {
            return true;
        }
    }

    /**
     * Delete account
     */
    public function deleteAccount($id)
    {
        $this->db->trans_start();
        
        $this->db->where('id', $id);
        $this->db->delete('accounts');
        
        $message = DELETE_RECORD_CONSTANT . " On accounts id " . $id;
        $action = "Delete";
        $this->log($message, $id, $action);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === false) {
            $this->db->trans_rollback();
            return false;
        } else {
            return true;
        }
    }

    /**
     * Get accounts by type
     */
    public function getAccountsByType($type)
    {
        $this->db->select('*');
        $this->db->from('accounts');
        $this->db->where('account_type', $type);
        $this->db->order_by('account_number', 'ASC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get total balance by account type
     */
    public function getTotalByType($type)
    {
        $this->db->select('SUM(opening_balance) as total');
        $this->db->from('accounts');
        $this->db->where('account_type', $type);
        $query = $this->db->get();
        $result = $query->row();
        return $result ? $result->total : 0;
    }

    /**
     * Get total income
     */
    public function getTotalIncome($date_from = null, $date_to = null)
    {
        $this->db->select('SUM(amount) as total');
        $this->db->from('income');
        
        if ($date_from && $date_to) {
            $this->db->where('date >=', $date_from);
            $this->db->where('date <=', $date_to);
        }
        
        $query = $this->db->get();
        $result = $query->row();
        return $result ? $result->total : 0;
    }

    /**
     * Get total expense
     */
    public function getTotalExpense($date_from = null, $date_to = null)
    {
        $this->db->select('SUM(amount) as total');
        $this->db->from('expenses');
        
        if ($date_from && $date_to) {
            $this->db->where('date >=', $date_from);
            $this->db->where('date <=', $date_to);
        }
        
        $query = $this->db->get();
        $result = $query->row();
        return $result ? $result->total : 0;
    }

    /**
     * Get income by category
     */
    public function getIncomeByCategory($date_from = null, $date_to = null)
    {
        $this->db->select('income_head.income_category, SUM(income.amount) as total');
        $this->db->from('income');
        $this->db->join('income_head', 'income.income_head_id = income_head.id');
        
        if ($date_from && $date_to) {
            $this->db->where('income.date >=', $date_from);
            $this->db->where('income.date <=', $date_to);
        }
        
        $this->db->group_by('income_head.income_category');
        $this->db->order_by('total', 'DESC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get expense by category
     */
    public function getExpenseByCategory($date_from = null, $date_to = null)
    {
        $this->db->select('expense_head.exp_category, SUM(expenses.amount) as total');
        $this->db->from('expenses');
        $this->db->join('expense_head', 'expenses.exp_head_id = expense_head.id');
        
        if ($date_from && $date_to) {
            $this->db->where('expenses.date >=', $date_from);
            $this->db->where('expenses.date <=', $date_to);
        }
        
        $this->db->group_by('expense_head.exp_category');
        $this->db->order_by('total', 'DESC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get monthly income trend
     */
    public function getMonthlyIncomeTrend($year = null)
    {
        if (!$year) {
            $year = date('Y');
        }
        
        $this->db->select('MONTH(date) as month, SUM(amount) as total');
        $this->db->from('income');
        $this->db->where('YEAR(date)', $year);
        $this->db->group_by('MONTH(date)');
        $this->db->order_by('month', 'ASC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get monthly expense trend
     */
    public function getMonthlyExpenseTrend($year = null)
    {
        if (!$year) {
            $year = date('Y');
        }
        
        $this->db->select('MONTH(date) as month, SUM(amount) as total');
        $this->db->from('expenses');
        $this->db->where('YEAR(date)', $year);
        $this->db->group_by('MONTH(date)');
        $this->db->order_by('month', 'ASC');
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get cash flow data
     */
    public function getCashFlow($date_from, $date_to)
    {
        $cash_flow = array();
        
        // Income transactions
        $this->db->select('date, amount, "income" as type, name as description');
        $this->db->from('income');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $income_query = $this->db->get_compiled_select();
        
        // Expense transactions
        $this->db->select('date, amount, "expense" as type, name as description');
        $this->db->from('expenses');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $expense_query = $this->db->get_compiled_select();
        
        // Union both queries
        $query = $this->db->query("($income_query) UNION ALL ($expense_query) ORDER BY date DESC");
        
        return $query->result_array();
    }

    /**
     * Get financial ratios
     */
    public function getFinancialRatios($date_from, $date_to)
    {
        $total_income = $this->getTotalIncome($date_from, $date_to);
        $total_expense = $this->getTotalExpense($date_from, $date_to);
        
        $ratios = array();
        
        // Profit margin
        if ($total_income > 0) {
            $ratios['profit_margin'] = (($total_income - $total_expense) / $total_income) * 100;
        } else {
            $ratios['profit_margin'] = 0;
        }
        
        // Expense ratio
        if ($total_income > 0) {
            $ratios['expense_ratio'] = ($total_expense / $total_income) * 100;
        } else {
            $ratios['expense_ratio'] = 0;
        }
        
        // Revenue growth (compared to previous period)
        $period_days = (strtotime($date_to) - strtotime($date_from)) / (60 * 60 * 24);
        $prev_date_from = date('Y-m-d', strtotime($date_from . " -$period_days days"));
        $prev_date_to = date('Y-m-d', strtotime($date_to . " -$period_days days"));
        
        $prev_income = $this->getTotalIncome($prev_date_from, $prev_date_to);
        
        if ($prev_income > 0) {
            $ratios['revenue_growth'] = (($total_income - $prev_income) / $prev_income) * 100;
        } else {
            $ratios['revenue_growth'] = 0;
        }
        
        return $ratios;
    }

    /**
     * Get account balance
     */
    public function getAccountBalance($account_id)
    {
        // Get opening balance
        $account = $this->getAccount($account_id);
        $opening_balance = $account ? $account['opening_balance'] : 0;
        
        // Calculate current balance based on transactions
        // This would need to be implemented based on your transaction tracking system
        
        return $opening_balance;
    }
}
