<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Verify extends MY_Addon_QRAttendanceController
{
    function __construct()
    {
        parent::__construct();
    }

    public function index()
    {
        if (!$this->rbac->hasPrivilege('face_attendance', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'faceattendance');
        $this->session->set_userdata('sub_menu', 'admin/faceattendance/verify/index');

        // Get all face data
        $data = array();
        $data['students'] = $this->getStudentsWithFaceData();
        $data['staff'] = $this->getStaffWithFaceData();
        
        $html = $this->getVerificationHTML($data);
        echo $html;
    }
    
    private function getStudentsWithFaceData()
    {
        // Check if the face_descriptor column exists
        $fields = $this->db->list_fields('students');
        if (!in_array('face_descriptor', $fields)) {
            return array();
        }
        
        // Get students with face data
        $this->db->select('id, admission_no, firstname, lastname, face_descriptor');
        $this->db->from('students');
        $this->db->where('face_descriptor IS NOT NULL');
        return $this->db->get()->result_array();
    }
    
    private function getStaffWithFaceData()
    {
        // Check if the face_descriptor column exists
        $fields = $this->db->list_fields('staff');
        if (!in_array('face_descriptor', $fields)) {
            return array();
        }
        
        // Get staff with face data
        $this->db->select('id, employee_id, name, surname, face_descriptor');
        $this->db->from('staff');
        $this->db->where('face_descriptor IS NOT NULL');
        return $this->db->get()->result_array();
    }
    
    private function getVerificationHTML($data)
    {
        $html = $this->load->view('layout/header', '', true);
        
        $html .= '
        <div class="content-wrapper">
            <section class="content">
                <div class="row">
                    <div class="col-md-12">
                        <div class="box box-primary">
                            <div class="box-header with-border">
                                <h3 class="box-title">Face Data Verification</h3>
                                <div class="box-tools pull-right">
                                    <button type="button" id="bulk-delete-btn" class="btn btn-danger btn-sm"><i class="fa fa-trash"></i> Delete All Face Data</button>
                                    <a href="' . site_url('admin/faceattendance/register') . '" class="btn btn-primary btn-sm">Register Face</a>
                                    <a href="' . site_url('admin/faceattendance/attendance') . '" class="btn btn-success btn-sm">Face Attendance</a>
                                </div>
                            </div>
                            <div class="box-body">
                                <div class="alert alert-info">
                                    This page shows all students and staff with registered face data. If you see entries here, it means face data is being stored correctly.
                                </div>
                                
                                <h4>Students with Face Data (' . count($data['students']) . ')</h4>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Admission No</th>
                                                <th>Name</th>
                                                <th>Face Data Status</th>
                                                <th>Face Data Sample</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>';
                                        
        if (!empty($data['students'])) {
            foreach ($data['students'] as $student) {
                $html .= '
                <tr>
                    <td>' . $student['id'] . '</td>
                    <td>' . $student['admission_no'] . '</td>
                    <td>' . $student['firstname'] . ' ' . $student['lastname'] . '</td>
                    <td><span class="label label-success">Available</span></td>
                    <td><small>' . substr($student['face_descriptor'], 0, 50) . '...</small></td>
                    <td>
                        <button type="button" class="btn btn-xs btn-warning edit-face-btn" data-id="' . $student['id'] . '" data-type="student" data-name="' . $student['firstname'] . ' ' . $student['lastname'] . '">
                            <i class="fa fa-edit"></i> Edit
                        </button>
                        <button type="button" class="btn btn-xs btn-danger delete-face-btn" data-id="' . $student['id'] . '" data-type="student" data-name="' . $student['firstname'] . ' ' . $student['lastname'] . '">
                            <i class="fa fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>';
            }
        } else {
            $html .= '<tr><td colspan="6" class="text-center">No students with face data found</td></tr>';
        }
        
        $html .= '
                                        </tbody>
                                    </table>
                                </div>
                                
                                <h4>Staff with Face Data (' . count($data['staff']) . ')</h4>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Employee ID</th>
                                                <th>Name</th>
                                                <th>Face Data Status</th>
                                                <th>Face Data Sample</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>';
                                        
        if (!empty($data['staff'])) {
            foreach ($data['staff'] as $staff) {
                $html .= '
                <tr>
                    <td>' . $staff['id'] . '</td>
                    <td>' . $staff['employee_id'] . '</td>
                    <td>' . $staff['name'] . '</td>
                    <td><span class="label label-success">Available</span></td>
                    <td><small>' . substr($staff['face_descriptor'], 0, 50) . '...</small></td>
                    <td>
                        <button type="button" class="btn btn-xs btn-warning edit-face-btn" data-id="' . $staff['id'] . '" data-type="staff" data-name="' . $staff['name'] . '">
                            <i class="fa fa-edit"></i> Edit
                        </button>
                        <button type="button" class="btn btn-xs btn-danger delete-face-btn" data-id="' . $staff['id'] . '" data-type="staff" data-name="' . $staff['name'] . '">
                            <i class="fa fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>';
            }
        } else {
            $html .= '<tr><td colspan="6" class="text-center">No staff with face data found</td></tr>';
        }
        
        $html .= '
                                        </tbody>
                                    </table>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-12">
                                        <div class="alert alert-warning">
                                            <h4>Troubleshooting</h4>
                                            <p>If you don\'t see any entries above:</p>
                                            <ol>
                                                <li>Make sure you have registered at least one face using the registration page</li>
                                                <li>Check if the face_descriptor column exists in the students and staff tables</li>
                                                <li>Check the server logs for any errors during face registration</li>
                                            </ol>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </div>';
        
        $html .= '
        <script>
        $(document).ready(function() {
            // Delete individual face data
            $(".delete-face-btn").click(function() {
                var id = $(this).data("id");
                var type = $(this).data("type");
                var name = $(this).data("name");
                
                if (confirm("Are you sure you want to delete face data for " + name + "?")) {
                    var button = $(this);
                    button.prop("disabled", true);
                    button.html("<i class=\"fa fa-spinner fa-spin\"></i> Deleting...");
                    
                    $.ajax({
                        url: "' . site_url('admin/faceattendance/verify/deleteFaceData') . '",
                        type: "POST",
                        data: {
                            id: id,
                            type: type
                        },
                        dataType: "json",
                        success: function(response) {
                            if (response.status === "success") {
                                button.closest("tr").fadeOut(500, function() {
                                    $(this).remove();
                                });
                                alert("Face data deleted successfully!");
                            } else {
                                alert("Error: " + response.message);
                                button.prop("disabled", false);
                                button.html("<i class=\"fa fa-trash\"></i> Delete");
                            }
                        },
                        error: function() {
                            alert("Error deleting face data. Please try again.");
                            button.prop("disabled", false);
                            button.html("<i class=\"fa fa-trash\"></i> Delete");
                        }
                    });
                }
            });
            
            // Edit face data
            $(".edit-face-btn").click(function() {
                var id = $(this).data("id");
                var type = $(this).data("type");
                var name = $(this).data("name");
                
                if (confirm("Do you want to re-register face data for " + name + "? This will redirect you to the registration page.")) {
                    // Redirect to registration page with edit parameters
                    window.location.href = "' . site_url('admin/faceattendance/register') . '?edit=" + type + "&id=" + id + "&name=" + encodeURIComponent(name);
                }
            });
            
            // Bulk delete all face data
            $("#bulk-delete-btn").click(function() {
                if (confirm("Are you sure you want to delete ALL face data for students and staff?")) {
                    if (confirm("This action cannot be undone. Are you absolutely sure?")) {
                        var button = $(this);
                        button.prop("disabled", true);
                        button.html("<i class=\"fa fa-spinner fa-spin\"></i> Deleting All...");
                        
                        $.ajax({
                            url: "' . site_url('admin/faceattendance/verify/deleteAllFaceData') . '",
                            type: "POST",
                            dataType: "json",
                            success: function(response) {
                                if (response.status === "success") {
                                    alert("All face data deleted successfully!");
                                    location.reload();
                                } else {
                                    alert("Error: " + response.message);
                                    button.prop("disabled", false);
                                    button.html("<i class=\"fa fa-trash\"></i> Delete All Face Data");
                                }
                            },
                            error: function() {
                                alert("Error deleting all face data. Please try again.");
                                button.prop("disabled", false);
                                button.html("<i class=\"fa fa-trash\"></i> Delete All Face Data");
                            }
                        });
                    }
                }
            });
        });
        </script>';
        
        $html .= $this->load->view('layout/footer', '', true);
        
        return $html;
    }
    
    public function deleteFaceData()
    {
        if (!$this->rbac->hasPrivilege('qr_code_attendance', 'can_delete')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }
        
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid parameters']);
            return;
        }
        
        try {
            if ($type === 'student') {
                $this->db->where('id', $id);
                $this->db->update('students', ['face_descriptor' => null]);
                $affected_rows = $this->db->affected_rows();
            } elseif ($type === 'staff') {
                $this->db->where('id', $id);
                $this->db->update('staff', ['face_descriptor' => null]);
                $affected_rows = $this->db->affected_rows();
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Invalid type']);
                return;
            }
            
            if ($affected_rows > 0) {
                echo json_encode(['status' => 'success', 'message' => 'Face data deleted successfully']);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'No face data found to delete']);
            }
        } catch (Exception $e) {
            error_log('Error deleting face data: ' . $e->getMessage());
            echo json_encode(['status' => 'error', 'message' => 'Database error occurred']);
        }
    }
    
    public function deleteAllFaceData()
    {
        if (!$this->rbac->hasPrivilege('qr_code_attendance', 'can_delete')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }
        
        try {
            // Delete all face data from students
            $this->db->update('students', ['face_descriptor' => null]);
            $students_affected = $this->db->affected_rows();
            
            // Delete all face data from staff
            $this->db->update('staff', ['face_descriptor' => null]);
            $staff_affected = $this->db->affected_rows();
            
            $total_affected = $students_affected + $staff_affected;
            
            if ($total_affected > 0) {
                echo json_encode([
                    'status' => 'success', 
                    'message' => "All face data deleted successfully. Removed face data from {$students_affected} students and {$staff_affected} staff members."
                ]);
            } else {
                echo json_encode(['status' => 'success', 'message' => 'No face data found to delete']);
            }
        } catch (Exception $e) {
            error_log('Error deleting all face data: ' . $e->getMessage());
            echo json_encode(['status' => 'error', 'message' => 'Database error occurred']);
        }
    }
}
