<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Attendance extends MY_Addon_QRAttendanceController
{
    function __construct()
    {
        parent::__construct();
        $this->load->model('qrattendance/qrsetting_model');
        $this->load->model('student_model');
        $this->load->model('staff_model');
        $this->load->model('smsconfig_model');
        $this->load->library('smsgateway');
        
        // Create SMS log table if it doesn't exist
        $this->createSmsLogTable();
    }

    public function index()
    {
        
        if (!$this->rbac->hasPrivilege('face_attendance', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'faceattendance');
        $this->session->set_userdata('sub_menu', 'admin/faceattendance/attendance/index');

        $this->load->library('media_storage');
        $data = array();
        $setting = $this->qrsetting_model->get();
        $data['setting'] = json_encode($setting);
        $data['version'] = $this->config->item('version');
        $data['page_title'] = 'Face Attendance';
        
        // Load the view file instead of direct HTML output
        try {
            $this->load->view('layout/header');
            $this->load->view('admin/faceattendance/index', $data);
            $this->load->view('layout/footer');
        } catch (Exception $e) {
            error_log('Error loading attendance page: ' . $e->getMessage());
            echo '<div class="alert alert-danger">Error loading attendance page. Please check error logs.</div>';
        }
    }
    
    public function checkout()
    {
        if (!$this->rbac->hasPrivilege('qr_code_attendance', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'faceattendance');
        $this->session->set_userdata('sub_menu', 'admin/faceattendance/attendance/checkout');

        $this->load->library('media_storage');
        $data = array();
        $setting = $this->qrsetting_model->get();
        $data['setting'] = json_encode($setting);
        $data['version'] = $this->config->item('version');
        $data['page_title'] = 'Face Checkout';
        
        // Load the checkout view file
        try {
            $this->load->view('layout/header');
            $this->load->view('admin/faceattendance/checkout', $data);
            $this->load->view('layout/footer');
        } catch (Exception $e) {
            error_log('Error loading checkout page: ' . $e->getMessage());
            echo '<div class="alert alert-danger">Error loading checkout page. Please check error logs.</div>';
        }
    }
    
    /**
     * Send SMS notification for attendance
     * 
     * @param int $id User ID
     * @param string $type User type (student/staff)
     * @return bool True if SMS sent successfully, false otherwise
     */
    public function sendAttendanceSMS($id, $type)
    {
        // Log the request
        error_log("Attendance Controller: Sending SMS for {$type} ID: {$id}");
        
        // Current time and date
        $time = date('h:i A');
        $date = date('d-m-Y');
        
        // Create SMS log table if it doesn't exist
        $this->createSmsLogTable();
        
        // Check if SMS already sent today
        $this->db->where('user_id', $id);
        $this->db->where('user_type', $type);
        $this->db->where('DATE(created_at)', date('Y-m-d'));
        $existing_sms = $this->db->get('face_attendance_sms_log')->row();
        
        if ($existing_sms) {
            error_log("Attendance Controller: SMS already sent today for {$type} ID: {$id}");
            return true; // Consider it a success if already sent
        }
        
        // Check SMS gateway
        $sms_detail = $this->smsconfig_model->getActiveSMS();
        if (empty($sms_detail)) {
            error_log("Attendance Controller: SMS gateway not configured");
            return false;
        }
        
        // Send SMS based on user type
        if ($type == 'student') {
            return $this->sendStudentSMS($id, $time, $date);
        } else if ($type == 'staff') {
            return $this->sendStaffSMS($id, $time, $date);
        }
        
        return false;
    }
    
    /**
     * Send SMS to student's guardian
     * 
     * @param int $student_id Student ID
     * @param string $time Attendance time
     * @param string $date Attendance date
     * @return bool True if SMS sent successfully, false otherwise
     */
    private function sendStudentSMS($student_id, $time, $date)
    {
        // Get student details
        $this->db->select('students.id, students.admission_no, students.firstname, students.middlename, students.lastname, students.guardian_phone, students.guardian_name');
        $this->db->from('students');
        $this->db->where('students.id', $student_id);
        $student = $this->db->get()->row();
        
        if (empty($student)) {
            error_log("Attendance Controller: Student not found with ID: {$student_id}");
            return false;
        }
        
        if (empty($student->guardian_phone)) {
            error_log("Attendance Controller: Guardian phone not available for student ID: {$student_id}");
            return false;
        }
        
        // Format student name
        $student_name = $student->firstname;
        if (!empty($student->middlename)) {
            $student_name .= ' ' . $student->middlename;
        }
        if (!empty($student->lastname)) {
            $student_name .= ' ' . $student->lastname;
        }
        
        // Prepare SMS message
        $message = "Dear {$student->guardian_name}, your ward {$student_name} ({$student->admission_no}) has successfully entered the school at {$time} on {$date}.";
        
        error_log("Attendance Controller: Sending SMS to {$student->guardian_phone} for student: {$student_name}");
        
        // Send SMS
        try {
            $result = $this->smsgateway->sendSMS($student->guardian_phone, $message, '', '');
            error_log("Attendance Controller: SMS sent to guardian. Result: " . json_encode($result));
            
            // Log SMS in database
            $sms_log = [
                'user_id' => $student_id,
                'user_type' => 'student',
                'phone' => $student->guardian_phone,
                'message' => $message,
                'status' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            $this->db->insert('face_attendance_sms_log', $sms_log);
            
            return true;
        } catch (Exception $e) {
            error_log("Attendance Controller: Error sending SMS: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send SMS to staff
     * 
     * @param int $staff_id Staff ID
     * @param string $time Attendance time
     * @param string $date Attendance date
     * @return bool True if SMS sent successfully, false otherwise
     */
    private function sendStaffSMS($staff_id, $time, $date)
    {
        // Get staff details
        $this->db->select('staff.id, staff.employee_id, staff.name, staff.surname, staff.contact_no');
        $this->db->from('staff');
        $this->db->where('staff.id', $staff_id);
        $staff = $this->db->get()->row();
        
        if (empty($staff)) {
            error_log("Attendance Controller: Staff not found with ID: {$staff_id}");
            return false;
        }
        
        if (empty($staff->contact_no)) {
            error_log("Attendance Controller: Phone number not available for staff ID: {$staff_id}");
            return false;
        }
        
        // Format staff name
        $staff_name = $staff->name;
        if (!empty($staff->surname)) {
            $staff_name .= ' ' . $staff->surname;
        }
        
        // Prepare SMS message
        $message = "Dear {$staff_name}, your attendance has been successfully marked at {$time} on {$date}.";
        
        error_log("Attendance Controller: Sending SMS to {$staff->contact_no} for staff: {$staff_name}");
        
        // Send SMS
        try {
            $result = $this->smsgateway->sendSMS($staff->contact_no, $message, '', '');
            error_log("Attendance Controller: SMS sent to staff. Result: " . json_encode($result));
            
            // Log SMS in database
            $sms_log = [
                'user_id' => $staff_id,
                'user_type' => 'staff',
                'phone' => $staff->contact_no,
                'message' => $message,
                'status' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            $this->db->insert('face_attendance_sms_log', $sms_log);
            
            return true;
        } catch (Exception $e) {
            error_log("Attendance Controller: Error sending SMS: " . $e->getMessage());
            return false;
        }
    }
    
    private function getFaceAttendanceHTML($data)
    {
        // HTML content for the face attendance page
        $html = '
        <div class="content-wrapper">
            <section class="content">
                <div class="row">
                    <div class="col-md-12">
                        <div class="box box-primary">
                            <div class="box-header ptbnull">
                                <h3 class="box-title"><i class="fa fa-camera"></i> ' . $this->lang->line('face_attendance') . '</h3>
                            </div>
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-8">
                                        <div class="tshadow mb25 bozero">
                                            <div class="box-header">
                                                <h4 class="box-title"><i class="fa fa-user-circle"></i> Face Recognition</h4>
                                            </div>
                                            <div class="box-body">
                                                <div class="row">
                                                    <div class="col-md-12 text-center">
                                                        <div id="face-detection-container" style="margin: 0 auto; width: 640px; height: 480px; border: 3px solid #3c8dbc; border-radius: 5px; overflow: hidden; background-color: #000;">
                                                            <video id="face-video" width="640" height="480" autoplay muted style="width: 100%; height: 100%; object-fit: cover;"></video>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="row" style="margin-top: 15px;">
                                                    <div class="col-md-12 text-center">
                                                        <button id="start-face-btn" class="btn btn-primary btn-lg" style="margin-right: 10px;"><i class="fa fa-play"></i> Start Face Recognition</button>
                                                        <button id="stop-face-btn" class="btn btn-danger btn-lg" style="display:none;"><i class="fa fa-stop"></i> Stop</button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="tshadow mb25 bozero">
                                            <div class="box-header">
                                                <h4 class="box-title"><i class="fa fa-id-card"></i> Manual Attendance</h4>
                                            </div>
                                            <div class="box-body">
                                                <div class="row">
                                                    <div class="col-md-12">
                                                        <div class="form-group">
                                                            <label for="user_type">Select User Type:</label>
                                                            <select id="user_type" class="form-control">
                                                                <option value="student">Student</option>
                                                                <option value="staff">Staff</option>
                                                            </select>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="row">
                                                    <div class="col-md-12">
                                                        <button id="id-attendance-btn" class="btn btn-success btn-block btn-lg"><i class="fa fa-check-circle"></i> Mark Attendance by ID</button>
                                                    </div>
                                                </div>
                                                <div class="row" style="margin-top: 15px;">
                                                    <div class="col-md-12">
                                                        <div id="manual-status" class="alert alert-info text-center">
                                                            Enter user ID and click "Mark Attendance"
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="tshadow mb25 bozero">
                                            <div class="box-header">
                                                <h4 class="box-title"><i class="fa fa-info-circle"></i> Instructions</h4>
                                            </div>
                                            <div class="box-body">
                                                <ol>
                                                    <li>Click "Start Face Recognition" to begin</li>
                                                    <li>Position your face in front of the camera</li>
                                                    <li>Only registered faces will be detected</li>
                                                    <li>Your name will be displayed when recognized</li>
                                                    <li>Attendance will be marked automatically</li>
                                                </ol>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </div>
        
        <!-- Load face-api.js and face-attendance.js -->
        <script src="' . base_url() . 'backend/faceattendance/face-api.min.js"></script>
        <link rel="stylesheet" href="' . base_url() . 'backend/faceattendance/face-api.css">
        <script src="' . base_url() . 'backend/faceattendance/face-attendance.js"></script>
        
        <script>
            $(document).ready(function() {
                // Simple direct attendance by ID
                $("#id-attendance-btn").on("click", function() {
                    var userId = prompt("Enter student/staff ID:");
                    var userType = $("#user_type").val() || "student";
                    
                    if (userId) {
                        // Show loading message
                        $("#manual-status").removeClass("alert-success alert-warning alert-danger").addClass("alert-info").text("Processing attendance for " + userType + " ID: " + userId);
                        
                        // Call the attendance API
                        $.ajax({
                            url: base_url + "admin/faceattendance/attendance/simpleMarkAttendance",
                            type: "POST",
                            data: {
                                id: userId,
                                type: userType
                            },
                            dataType: "json",
                            success: function(response) {
                                if (response.status) {
                                    if (response.already_marked) {
                                        $("#manual-status").removeClass("alert-info").addClass("alert-warning").text(response.msg);
                                    } else {
                                        $("#manual-status").removeClass("alert-info").addClass("alert-success").text(response.msg);
                                    }
                                } else {
                                    $("#manual-status").removeClass("alert-info").addClass("alert-danger").text(response.msg);
                                }
                            },
                            error: function(xhr, status, error) {
                                $("#manual-status").removeClass("alert-info").addClass("alert-danger").text("Error: " + error);
                            }
                        });
                    }
                });
            });
            
            var obj = jQuery.parseJSON(\'' . $data['setting'] . '\');
            var camera_type = obj.camera_type;
            var auto_attendance = obj.auto_attendance;
            var statusMatched = "' . $this->lang->line("matched") . '";
            var statusScanning = "' . $this->lang->line("scanning") . '";
            var base_url = "' . base_url() . '";
            
            // QR Code Scanner Setup
            var x = document.getElementById("successAudio");
            var lastResult, modalOpen = 0;
            const html5QrCode = new Html5Qrcode("reader");
            const qrCodeSuccessCallback = (decodedText, decodedResult) => {
                if (decodedText !== lastResult && modalOpen == 0) {
                    lastResult = decodedText;
                    modalOpen = 1;
                    var qrcode_value = decodedText;
                    $.ajax({
                        url: base_url + "admin/qrattendance/checkqr",
                        type: "POST",
                        data: {
                            qrcode: qrcode_value
                        },
                        dataType: \'json\',
                        beforeSend: function() {
                            $("#qrcodesuccess").html("");
                            $("#qrcodefail").html("");
                        },
                        success: function(data) {
                            if (data.status == "0") {
                                $("#qrcodefail").html(data.message);
                                x.play();
                                setTimeout(function() {
                                    modalOpen = 0;
                                }, 1000);
                            } else {
                                $("#qrcodesuccess").html(data.message);
                                x.play();
                                $("#profile_data").html(data.page);
                                $(".scanner").removeClass("displayblock").addClass("displaynone");
                                $("#qrbutton").removeClass("displaynone").addClass("displayblock");
                            }
                        },
                        error: function(jqXHR, textStatus, errorThrown) {}
                    });
                }
            };
            const config = {
                fps: 10,
                qrbox: {
                    width: 250,
                    height: 250
                }
            };
        </script>
        
        <!-- Face API CSS and JS -->
        <link href="' . base_url() . 'backend/faceattendance/face-api.css" rel="stylesheet">
        <script src="' . base_url() . 'backend/faceattendance/face-api.min.js"></script>
        <script src="' . base_url() . 'backend/faceattendance/face-attendance.js"></script>
        <script src="' . base_url() . 'backend/faceattendance/face-sms.js"></script>
        
        <!-- Additional CSS for better styling -->
        <style>
            #face-detection-container {
                position: relative;
                margin: 0 auto;
                width: 640px;
                height: 480px;
                border: 3px solid #3c8dbc;
                border-radius: 5px;
                overflow: hidden;
                background-color: #000;
                box-shadow: 0 4px 8px rgba(0,0,0,0.2);
            }
            
            #face-video {
                width: 100%;
                height: 100%;
                object-fit: cover;
            }
            
            #face-detection-container canvas {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
            }
            
            .btn-lg {
                padding: 10px 20px;
                font-size: 16px;
            }
            
            .box-title {
                font-weight: bold;
                color: #3c8dbc;
            }
            
            .alert {
                padding: 10px;
                border-radius: 4px;
                margin-top: 15px;
            }
            
            .tshadow {
                box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            }
            
            ol li {
                margin-bottom: 8px;
            }
        </style>
        ';
        
        return $html;
    }
    
    public function getFaceData()
    {
        // Get face data for all students and staff
        $this->db->select('students.id, "student" as type, students.face_descriptor as faceDescriptor, CONCAT(students.firstname, " ", COALESCE(students.middlename, ""), " ", COALESCE(students.lastname, "")) as name');
        $this->db->from('students');
        $this->db->where('students.face_descriptor IS NOT NULL');
        $this->db->where('students.face_descriptor != ""');
        $students = $this->db->get()->result();
        
        $this->db->select('staff.id, "staff" as type, staff.face_descriptor as faceDescriptor, CONCAT(staff.name, " ", COALESCE(staff.surname, "")) as name');
        $this->db->from('staff');
        $this->db->where('staff.face_descriptor IS NOT NULL');
        $this->db->where('staff.face_descriptor != ""');
        $staff = $this->db->get()->result();
        
        $faceData = array_merge($students, $staff);
        
        // Clean up the names (remove extra spaces)
        foreach ($faceData as $key => $data) {
            $faceData[$key]->name = trim(preg_replace('/\s+/', ' ', $data->name));
        }
        
        echo json_encode(['status' => true, 'faceData' => $faceData]);
    }
    
    public function markAttendance()
    {
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        $date = date('Y-m-d');
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => false, 'msg' => 'Invalid input data']);
            return;
        }
        
        if ($type == 'student') {
            // Get student details
            $student = $this->getStudentDetails($id);
            
            if (!$student) {
                echo json_encode(['status' => false, 'msg' => 'Student not found']);
                return;
            }
            
            // Check if attendance already marked
            $this->db->where('student_session_id', $student->student_session_id);
            $this->db->where('date', $date);
            $query = $this->db->get('student_attendences');
            
            if ($query->num_rows() > 0) {
                $response = array('status' => true, 'msg' => 'Attendance already marked for today');
            } else {
                // Mark attendance
                $biometric_device_data = [
                    "uid" => '',
                    "name" => '',
                    "user_type" => 'student',
                    "user_id" => $student->id,
                    "biometric_attendance_type" => 'face',
                    "date" => $date,
                    "created_at" => date('Y-m-d H:i:s'),
                ];
                
                $this->db->insert('biometric_attendance_log', $biometric_device_data);
                
                $attendance_data = [
                    "student_session_id" => $student->student_session_id,
                    "biometric_attendence" => 1,
                    "date" => $date,
                    "attendence_type_id" => 1,
                    "created_at" => date('Y-m-d H:i:s'),
                ];
                
                $this->db->insert('student_attendences', $attendance_data);
                $attendance_id = $this->db->insert_id();
                
                $student->attendance_id = $attendance_id;
                $data['student'] = $student;
                $data['status'] = 1;
                $data['profile_type'] = 'student';
                $page = $this->load->view('admin/qrattendance/_getProfileDetail', $data, true);
                $response = array('status' => true, 'page' => $page, 'msg' => 'Attendance marked successfully');
            }
            
            echo json_encode($response);
        } else if ($type == 'staff') {
            // Get staff details
            $staff = $this->getStaffDetails($id);
            
            if (!$staff) {
                echo json_encode(['status' => false, 'msg' => 'Staff not found']);
                return;
            }
            
            // Check if attendance already marked
            $this->db->where('staff_id', $staff->id);
            $this->db->where('date', $date);
            $query = $this->db->get('staff_attendance');
            
            if ($query->num_rows() > 0) {
                // Attendance already marked
                $staff->attendance_id = $query->row()->id;
                $data['staff'] = $staff;
                $data['status'] = 2;
                $data['profile_type'] = 'staff';
                $page = $this->load->view('admin/qrattendance/_getProfileDetail', $data, true);
                echo json_encode(['status' => true, 'page' => $page, 'msg' => 'Attendance already marked']);
            } else {
                // Mark attendance
                $biometric_device_data = [
                    "uid" => '',
                    "name" => '',
                    "user_type" => 'staff',
                    "user_id" => $staff->id,
                    "biometric_attendance_type" => 'face',
                    "date" => $date,
                    "created_at" => date('Y-m-d H:i:s'),
                ];
                
                $this->db->insert('biometric_attendance_log', $biometric_device_data);
                
                $attendance_data = [
                    "staff_id" => $staff->id,
                    "biometric_attendence" => 1,
                    "date" => $date,
                    "staff_attendance_type_id" => 1,
                    "created_at" => date('Y-m-d H:i:s'),
                ];
                
                $this->db->insert('staff_attendance', $attendance_data);
                $attendance_id = $this->db->insert_id();
                
                $staff->attendance_id = $attendance_id;
                $data['staff'] = $staff;
                $data['status'] = 1;
                $data['profile_type'] = 'staff';
                $page = $this->load->view('admin/qrattendance/_getProfileDetail', $data, true);
                echo json_encode(['status' => true, 'page' => $page, 'msg' => 'Attendance marked successfully']);
            }
        } else {
            echo json_encode(['status' => false, 'msg' => 'Invalid user type']);
        }
    }
    
    public function simpleMarkAttendance()
    {
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        $is_checkout = $this->input->post('is_checkout') ? $this->input->post('is_checkout') : 0;
        $date = date('Y-m-d');
        
        // Debug information
        error_log("Attendance marking request - ID: {$id}, Type: {$type}, Is Checkout: {$is_checkout}, Date: {$date}");
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => false, 'msg' => 'Invalid input data']);
            return;
        }
        
        // Check time validation for check-in
        if (!$is_checkout) {
            $time_validation = $this->validateCheckinTime();
            if (!$time_validation['allowed']) {
                echo json_encode(['status' => false, 'msg' => $time_validation['message']]);
                return;
            }
        }
        
        if ($type == 'student') {
            // Get student details with better error handling
            $this->db->select('student_session.id as student_session_id, students.id, students.admission_no, students.firstname, students.middlename, students.lastname, students.image');
            $this->db->from('students');
            $this->db->join('student_session', 'student_session.student_id = students.id', 'left');
            $this->db->where('students.id', $id);
            
            // Only apply session filter if we have a valid session ID
            if (!empty($this->current_session)) {
                $this->db->where('student_session.session_id', $this->current_session);
            }
            
            $student = $this->db->get()->row();
            
            // Debug query
            error_log("Student query: " . $this->db->last_query());
            error_log("Student found: " . ($student ? 'Yes' : 'No'));
            
            if (!$student) {
                echo json_encode(['status' => false, 'msg' => 'Student not found']);
                return;
            }
            
            // Check if attendance already marked
            $this->db->where('student_session_id', $student->student_session_id);
            $this->db->where('date', $date);
            $attendance = $this->db->get('student_attendences')->row();
            
            if ($attendance) {
                // Attendance already marked
                echo json_encode(['status' => true, 'already_marked' => true, 'msg' => 'Attendance already marked for today']);
            } else {
                // Mark attendance
                // Check if we have a valid student_session_id
                if (empty($student->student_session_id)) {
                    // Get the most recent student session for this student
                    $this->db->select('student_session.id as student_session_id');
                    $this->db->from('student_session');
                    $this->db->where('student_session.student_id', $student->id);
                    $this->db->order_by('student_session.id', 'DESC');
                    $this->db->limit(1);
                    $session_result = $this->db->get()->row();
                    
                    error_log("Looking for student session - Query: " . $this->db->last_query());
                    
                    if ($session_result) {
                        $student->student_session_id = $session_result->student_session_id;
                        error_log("Found student_session_id: {$student->student_session_id}");
                    } else {
                        // Create a temporary session ID if none exists
                        // This ensures attendance can still be marked
                        error_log("No student session found, using direct student ID");
                        $student->student_session_id = $student->id;
                    }
                }
                
                // Get additional student details for complete record
                $this->db->select('classes.class, sections.section');
                $this->db->from('student_session');
                $this->db->join('classes', 'classes.id = student_session.class_id', 'left');
                $this->db->join('sections', 'sections.id = student_session.section_id', 'left');
                $this->db->where('student_session.id', $student->student_session_id);
                $class_details = $this->db->get()->row();
                
                error_log("Class details query: " . $this->db->last_query());
                error_log("Class details found: " . ($class_details ? 'Yes' : 'No'));
                
                // Create more detailed attendance record
                $attendance_data = [
                    'student_session_id' => $student->student_session_id,
                    'attendence_type_id' => 1, // Present
                    'date' => $date,
                    'created_at' => date('Y-m-d H:i:s'),
                    'remark' => 'Marked via face recognition',
                    'biometric_attendence' => 1
                ];
                
                error_log("Inserting attendance data: " . json_encode($attendance_data));
                $this->db->insert('student_attendences', $attendance_data);
                error_log("Insert result: " . $this->db->affected_rows() . " rows affected");
                
                if ($this->db->affected_rows() > 0) {
                    // Attendance marked successfully, now send SMS automatically
                    $sms_result = $this->sendAttendanceSMS($student->id, 'student');
                    error_log("Auto SMS for student ID {$student->id}: " . ($sms_result ? 'Sent' : 'Failed'));
                    
                    echo json_encode(['status' => true, 'already_marked' => false, 'msg' => 'Attendance marked successfully and SMS notification sent']);
                } else {
                    echo json_encode(['status' => false, 'msg' => 'Failed to mark attendance']);
                }
            }
        } else if ($type == 'staff') {
            // Get staff details
            $this->db->select('staff.id, staff.employee_id, staff.name, staff.surname, staff.image');
            $this->db->where('staff.id', $id);
            $staff = $this->db->get('staff')->row();
            
            if (!$staff) {
                echo json_encode(['status' => false, 'msg' => 'Staff not found']);
                return;
            }
            
            // Check if attendance already marked
            $this->db->where('staff_id', $staff->id);
            $this->db->where('date', $date);
            $attendance = $this->db->get('staff_attendance')->row();
            
            if ($attendance) {
                // Attendance already marked
                echo json_encode(['status' => true, 'already_marked' => true, 'msg' => 'Attendance already marked for today']);
            } else {
                // Mark attendance
                // Get additional staff details for complete record
                $this->db->select('staff_designation.designation, department.department_name, roles.name as role');
                $this->db->from('staff');
                $this->db->join('staff_designation', 'staff_designation.id = staff.designation', 'left');
                $this->db->join('department', 'department.id = staff.department', 'left');
                $this->db->join('staff_roles', 'staff_roles.staff_id = staff.id', 'left');
                $this->db->join('roles', 'roles.id = staff_roles.role_id', 'left');
                $this->db->where('staff.id', $staff->id);
                $staff_details = $this->db->get()->row();
                
                // Create more detailed attendance record
                $attendance_data = [
                    'staff_id' => $staff->id,
                    'staff_attendance_type_id' => 1, // Present
                    'date' => $date,
                    'created_at' => date('Y-m-d H:i:s'),
                    'remark' => 'Marked via face recognition',
                    'biometric_attendence' => 1
                ];
                
                $this->db->insert('staff_attendance', $attendance_data);
                
                if ($this->db->affected_rows() > 0) {
                    // Attendance marked successfully, now send SMS automatically
                    $sms_result = $this->sendAttendanceSMS($staff->id, 'staff');
                    error_log("Auto SMS for staff ID {$staff->id}: " . ($sms_result ? 'Sent' : 'Failed'));
                    
                    echo json_encode(['status' => true, 'already_marked' => false, 'msg' => 'Attendance marked successfully and SMS notification sent']);
                } else {
                    echo json_encode(['status' => false, 'msg' => 'Failed to mark attendance']);
                }
            }
        } else {
            echo json_encode(['status' => false, 'msg' => 'Invalid user type']);
        }
    }
    
    private function getStudentDetails($id)
    {
        $this->db->select('students.id, students.admission_no, students.firstname, students.middlename, students.lastname, students.father_name, students.mother_name, students.guardian_name, students.guardian_relation, students.guardian_phone, students.local_guardian_name, students.local_guardian_phone, students.local_guardian_email, students.local_guardian_address, students.gender, students.dob, students.mobileno, students.email, students.religion, students.cast, students.house_name, students.roll_no, students.admission_date, students.rte, students.image, students.mode_of_payment, students.enrollment_type, students.middlename, students.lastname, students.guardian_email, students.current_address, students.permanent_address, students.route_id, students.school_house_id, students.blood_group, students.vehroute_id, students.hostel_room_id, students.adhar_no, students.samagra_id, students.bank_account_no, students.bank_name, students.ifsc_code, students.guardian_occupation, students.guardian_address, students.is_active, students.previous_school, students.height, students.weight, students.measurement_date, students.app_key, students.parent_app_key, students.disable_at, students.note, students.dis_reason, students.disable_reason_date, students.app_key, students.parent_app_key, students.created_at, students.updated_at, student_session.id as student_session_id, student_session.session_id, student_session.student_id, student_session.class_id, student_session.section_id, student_session.vehroute_id, student_session.hostel_room_id, student_session.transport_fees, student_session.fees_discount, student_session.is_active, student_session.is_alumni, student_session.default_login, student_session.created_at, student_session.updated_at, classes.id as class_id, classes.class, sections.id as section_id, sections.section');
        $this->db->from('students');
        $this->db->join('student_session', 'student_session.student_id = students.id');
        $this->db->join('classes', 'student_session.class_id = classes.id');
        $this->db->join('sections', 'sections.id = student_session.section_id');
        $this->db->where('students.id', $id);
        $this->db->where('student_session.session_id', $this->current_session);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row();
        }
        
        return false;
    }
    
    
    
    private function getStaffDetails($id)
    {
        $this->db->select('staff.id, staff.employee_id, staff.name, staff.surname, staff.contact_no, staff.emergency_contact_no, staff.email, staff.dob, staff.marital_status, staff.date_of_joining, staff.image, staff.designation, staff.qualification, staff.work_exp, staff.address, staff_designation.designation as designation_name, department.department_name, roles.name as role_name');
        $this->db->from('staff');
        $this->db->join('staff_designation', 'staff_designation.id = staff.designation', 'left');
        $this->db->join('department', 'department.id = staff.department', 'left');
        $this->db->join('staff_roles', 'staff_roles.staff_id = staff.id', 'left');
        $this->db->join('roles', 'roles.id = staff_roles.role_id', 'left');
        $this->db->where('staff.id', $id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row();
        }
        
        return false;
    }
    
    /**
     * Get attendance report for students and staff
     */
    public function getAttendanceReport()
    {
        $date = $this->input->get('date');
        if (empty($date)) {
            $date = date('Y-m-d');
        }
        
        $data = [];
        $data['date'] = $date;
        $data['title'] = 'Attendance Report for ' . date('d M Y', strtotime($date));
        
        // Get student attendance with proper in-time and out-time
        // Explicitly select created_at for in-time and out_time for out-time to avoid confusion
        $this->db->select('student_attendences.id, student_attendences.student_session_id, student_attendences.biometric_attendence, 
                          student_attendences.date, student_attendences.created_at, student_attendences.updated_at, 
                          student_attendences.out_time, student_attendences.has_checked_out, student_attendences.remark, 
                          students.id as student_id, students.admission_no, students.firstname, students.middlename, 
                          students.lastname, classes.class, sections.section');
        $this->db->from('student_attendences');
        $this->db->join('student_session', 'student_session.id = student_attendences.student_session_id');
        $this->db->join('students', 'students.id = student_session.student_id');
        $this->db->join('classes', 'classes.id = student_session.class_id');
        $this->db->join('sections', 'sections.id = student_session.section_id');
        $this->db->where('student_attendences.date', $date);
        $this->db->order_by('classes.class', 'ASC');
        $this->db->order_by('sections.section', 'ASC');
        $this->db->order_by('students.firstname', 'ASC');
        $data['student_attendance'] = $this->db->get()->result();
        
        // Debug output
        error_log("Found " . count($data['student_attendance']) . " student attendance records for date: {$date}");
        
        // Debug the actual values of created_at and out_time for student attendance
        if (!empty($data['student_attendance'])) {
            foreach ($data['student_attendance'] as $index => $attendance) {
                error_log("Student {$index} - ID: {$attendance->student_id}, created_at: {$attendance->created_at}, out_time: {$attendance->out_time}");
            }
        }
        
        // Get staff attendance with proper in-time and out-time
        // Explicitly select created_at for in-time and out_time for out-time to avoid confusion
        $this->db->select('staff_attendance.id, staff_attendance.staff_id, staff_attendance.biometric_attendence, 
                          staff_attendance.date, staff_attendance.created_at, staff_attendance.updated_at, 
                          staff_attendance.out_time, staff_attendance.has_checked_out, staff_attendance.remark, 
                          staff.name, staff.surname, staff.employee_id, staff_designation.designation');
        $this->db->from('staff_attendance');
        $this->db->join('staff', 'staff.id = staff_attendance.staff_id');
        $this->db->join('staff_designation', 'staff_designation.id = staff.designation', 'left');
        $this->db->where('staff_attendance.date', $date);
        $this->db->order_by('staff.name', 'ASC');
        $data['staff_attendance'] = $this->db->get()->result();
        
        $this->load->view('layout/header');
        $this->load->view('admin/faceattendance/attendance_report', $data);
        $this->load->view('layout/footer');
    }
    
    /**
     * Send SMS from attendance report page
     * 
     * This method is called via AJAX from the attendance report page
     * to send SMS notifications to students/staff
     */
    
    /**
     * Create face_attendance_sms_log table if it doesn't exist
     */
    private function createSmsLogTable()
    {
        // Check if the table exists
        if (!$this->db->table_exists('face_attendance_sms_log')) {
            try {
                // Create the table
                $this->db->query("
                    CREATE TABLE `face_attendance_sms_log` (
                      `id` int(11) NOT NULL AUTO_INCREMENT,
                      `user_id` int(11) NOT NULL,
                      `user_type` varchar(20) NOT NULL,
                      `phone` varchar(20) NOT NULL,
                      `message` text NOT NULL,
                      `status` tinyint(1) NOT NULL DEFAULT '1',
                      `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                      PRIMARY KEY (`id`),
                      KEY `user_id` (`user_id`),
                      KEY `user_type` (`user_type`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
                ");
                
                error_log("Attendance Controller: Created face_attendance_sms_log table");
            } catch (Exception $e) {
                error_log("Attendance Controller: Error creating face_attendance_sms_log table: " . $e->getMessage());
            }
        }
    }
    
    public function sendSmsFromReport()
    {
        // Get parameters
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        $time = $this->input->post('time');
        $date = $this->input->post('date');
        
        // Log request for debugging
        error_log("Attendance Controller: Sending SMS from report - ID: {$id}, Type: {$type}, Time: {$time}, Date: {$date}");
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => false, 'msg' => 'Missing required parameters']);
            return;
        }
        
        // Check SMS gateway
        $sms_detail = $this->smsconfig_model->getActiveSMS();
        if (empty($sms_detail)) {
            echo json_encode(['status' => false, 'msg' => 'SMS gateway not configured']);
            return;
        }
        
        // Create SMS log table if it doesn't exist
        $this->createSmsLogTable();
        
        // Check if SMS already sent today
        $this->db->where('user_id', $id);
        $this->db->where('user_type', $type);
        $this->db->where('DATE(created_at)', date('Y-m-d'));
        $existing_sms = $this->db->get('face_attendance_sms_log')->row();
        
        if ($existing_sms) {
            echo json_encode(['status' => true, 'msg' => 'SMS already sent today']);
            return;
        }
        
        // Send SMS based on user type
        if ($type == 'student') {
            // Get student details
            $this->db->select('students.id, students.admission_no, students.firstname, students.middlename, students.lastname, students.guardian_phone, students.guardian_name');
            $this->db->from('students');
            $this->db->where('students.id', $id);
            $student = $this->db->get()->row();
            
            if (empty($student)) {
                echo json_encode(['status' => false, 'msg' => 'Student not found']);
                return;
            }
            
            if (empty($student->guardian_phone)) {
                echo json_encode(['status' => false, 'msg' => 'Guardian phone not available']);
                return;
            }
            
            // Format student name
            $student_name = $student->firstname;
            if (!empty($student->middlename)) {
                $student_name .= ' ' . $student->middlename;
            }
            if (!empty($student->lastname)) {
                $student_name .= ' ' . $student->lastname;
            }
            
            // Prepare SMS message
            $message = "Dear {$student->guardian_name}, your ward {$student_name} ({$student->admission_no}) has successfully entered the school at {$time} on {$date}.";
            
            error_log("Attendance Controller: Sending SMS to {$student->guardian_phone} for student: {$student_name}");
            
            // Send SMS
            try {
                $result = $this->smsgateway->sendSMS($student->guardian_phone, $message, '', '');
                error_log("Attendance Controller: SMS sent to guardian. Result: " . json_encode($result));
                
                // Log SMS in database
                $sms_log = [
                    'user_id' => $student->id,
                    'user_type' => 'student',
                    'phone' => $student->guardian_phone,
                    'message' => $message,
                    'status' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                $this->db->insert('face_attendance_sms_log', $sms_log);
                
                echo json_encode(['status' => true, 'msg' => 'SMS notification sent to guardian']);
            } catch (Exception $e) {
                error_log("Attendance Controller: Error sending SMS: " . $e->getMessage());
                echo json_encode(['status' => false, 'msg' => 'Error sending SMS: ' . $e->getMessage()]);
            }
        } else if ($type == 'staff') {
            // Get staff details
            $this->db->select('staff.id, staff.employee_id, staff.name, staff.surname, staff.contact_no');
            $this->db->from('staff');
            $this->db->where('staff.id', $id);
            $staff = $this->db->get()->row();
            
            if (empty($staff)) {
                echo json_encode(['status' => false, 'msg' => 'Staff not found']);
                return;
            }
            
            if (empty($staff->contact_no)) {
                echo json_encode(['status' => false, 'msg' => 'Staff phone number not available']);
                return;
            }
            
            // Format staff name
            $staff_name = $staff->name;
            if (!empty($staff->surname)) {
                $staff_name .= ' ' . $staff->surname;
            }
            
            // Prepare SMS message
            $message = "Dear {$staff_name}, your attendance has been successfully marked at {$time} on {$date}.";
            
            error_log("Attendance Controller: Sending SMS to {$staff->contact_no} for staff: {$staff_name}");
            
            // Send SMS
            try {
                $result = $this->smsgateway->sendSMS($staff->contact_no, $message, '', '');
                error_log("Attendance Controller: SMS sent to staff. Result: " . json_encode($result));
                
                // Log SMS in database
                $sms_log = [
                    'user_id' => $staff->id,
                    'user_type' => 'staff',
                    'phone' => $staff->contact_no,
                    'message' => $message,
                    'status' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                $this->db->insert('face_attendance_sms_log', $sms_log);
                
                echo json_encode(['status' => true, 'msg' => 'SMS notification sent to staff']);
            } catch (Exception $e) {
                error_log("Attendance Controller: Error sending SMS: " . $e->getMessage());
                echo json_encode(['status' => false, 'msg' => 'Error sending SMS: ' . $e->getMessage()]);
            }
        } else {
            echo json_encode(['status' => false, 'msg' => 'Invalid user type']);
        }
    }
    
    public function getAttendanceReportData()
    {
        // Initialize attendance data array
        $attendance_data = [];
        
        // Get attendance data from database
        $this->db->select('*');
        $this->db->from('face_attendance');
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $attendance_data = $query->result_array();
        }
        
        echo json_encode(['status' => true, 'data' => $attendance_data]);
    }
    
    public function getUserName()
    {
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => false, 'msg' => 'Invalid input data']);
            return;
        }
        
        if ($type == 'student') {
            // Get student name
            $this->db->select('CONCAT(students.firstname, " ", COALESCE(students.middlename, ""), " ", COALESCE(students.lastname, "")) as name');
            $this->db->from('students');
            $this->db->where('students.id', $id);
            $student = $this->db->get()->row();
            
            if ($student) {
                echo json_encode(['status' => true, 'name' => trim($student->name)]);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Student not found']);
            }
        } else if ($type == 'staff') {
            // Get staff name
            $this->db->select('CONCAT(staff.name, " ", COALESCE(staff.surname, "")) as name');
            $this->db->from('staff');
            $this->db->where('staff.id', $id);
            $staff = $this->db->get()->row();
            
            if ($staff) {
                echo json_encode(['status' => true, 'name' => trim($staff->name)]);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Staff not found']);
            }
        } else {
            echo json_encode(['status' => false, 'msg' => 'Invalid user type']);
        }
    }
    
    /**
     * Delete attendance record
     * 
     * This method handles the deletion of attendance records for both students and staff
     * It is called via AJAX from the attendance report page
     */
    public function deleteAttendance()
    {
        // Set content type for JSON response
        header('Content-Type: application/json');
        
        // Log all incoming data for debugging
        error_log("Delete Attendance Request - POST data: " . json_encode($_POST));
        error_log("Delete Attendance Request - Input data: " . json_encode($this->input->post()));
        
        // Check if user has permission to delete attendance (temporarily bypass for debugging)
        $has_permission = true; // Temporarily set to true for debugging
        try {
            $has_permission = $this->rbac->hasPrivilege('qr_code_attendance', 'can_delete');
        } catch (Exception $e) {
            error_log("Permission check error: " . $e->getMessage());
            // Continue with true for debugging
        }
        
        if (!$has_permission) {
            error_log("Permission denied for delete attendance");
            echo json_encode(['status' => false, 'msg' => 'You do not have permission to delete attendance records']);
            return;
        }
        
        // Get parameters
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        
        // Log request for debugging
        error_log("Attendance Controller: Deleting attendance - ID: {$id}, Type: {$type}");
        
        if (empty($id) || empty($type)) {
            error_log("Missing parameters - ID: " . var_export($id, true) . ", Type: " . var_export($type, true));
            echo json_encode(['status' => false, 'msg' => 'Missing required parameters (ID: ' . $id . ', Type: ' . $type . ')']);
            return;
        }
        
        // Delete attendance record based on type
        if ($type == 'student') {
            // First check if record exists
            $this->db->where('id', $id);
            $existing = $this->db->get('student_attendences')->row();
            
            if (!$existing) {
                error_log("Student attendance record not found with ID: {$id}");
                echo json_encode(['status' => false, 'msg' => 'Student attendance record not found']);
                return;
            }
            
            // Delete student attendance record
            $this->db->where('id', $id);
            $result = $this->db->delete('student_attendences');
            
            if ($result && $this->db->affected_rows() > 0) {
                // Log the deletion
                error_log("Attendance Controller: Successfully deleted student attendance record ID: {$id}");
                echo json_encode(['status' => true, 'msg' => 'Student attendance record deleted successfully']);
            } else {
                error_log("Attendance Controller: Failed to delete student attendance record ID: {$id}, affected rows: " . $this->db->affected_rows());
                echo json_encode(['status' => false, 'msg' => 'Failed to delete student attendance record']);
            }
        } else if ($type == 'staff') {
            // First check if record exists
            $this->db->where('id', $id);
            $existing = $this->db->get('staff_attendance')->row();
            
            if (!$existing) {
                error_log("Staff attendance record not found with ID: {$id}");
                echo json_encode(['status' => false, 'msg' => 'Staff attendance record not found']);
                return;
            }
            
            // Delete staff attendance record
            $this->db->where('id', $id);
            $result = $this->db->delete('staff_attendance');
            
            if ($result && $this->db->affected_rows() > 0) {
                // Log the deletion
                error_log("Attendance Controller: Successfully deleted staff attendance record ID: {$id}");
                echo json_encode(['status' => true, 'msg' => 'Staff attendance record deleted successfully']);
            } else {
                error_log("Attendance Controller: Failed to delete staff attendance record ID: {$id}, affected rows: " . $this->db->affected_rows());
                echo json_encode(['status' => false, 'msg' => 'Failed to delete staff attendance record']);
            }
        } else {
            error_log("Invalid user type: {$type}");
            echo json_encode(['status' => false, 'msg' => 'Invalid user type: ' . $type]);
        }
    }
    
    /**
     * Delete all attendance records for a specific date
     * 
     * This method handles the bulk deletion of all attendance records (both students and staff) for a given date
     * It is called via AJAX from the attendance report page
     */
    public function deleteAllAttendance()
    {
        // Set content type for JSON response
        header('Content-Type: application/json');
        
        // Log all incoming data for debugging
        error_log("Delete All Attendance Request - POST data: " . json_encode($_POST));
        
        // Check if user has permission to delete attendance (temporarily bypass for debugging)
        $has_permission = true; // Temporarily set to true for debugging
        try {
            $has_permission = $this->rbac->hasPrivilege('qr_code_attendance', 'can_delete');
        } catch (Exception $e) {
            error_log("Permission check error: " . $e->getMessage());
            // Continue with true for debugging
        }
        
        if (!$has_permission) {
            error_log("Permission denied for delete all attendance");
            echo json_encode(['status' => false, 'msg' => 'You do not have permission to delete attendance records']);
            return;
        }
        
        // Get parameters
        $date = $this->input->post('date');
        
        // Log request for debugging
        error_log("Attendance Controller: Deleting all attendance for date: {$date}");
        
        if (empty($date)) {
            error_log("Missing date parameter");
            echo json_encode(['status' => false, 'msg' => 'Missing required date parameter']);
            return;
        }
        
        // Count records before deletion
        $this->db->where('date', $date);
        $student_count = $this->db->count_all_results('student_attendences');
        
        $this->db->where('date', $date);
        $staff_count = $this->db->count_all_results('staff_attendance');
        
        $total_count = $student_count + $staff_count;
        
        if ($total_count == 0) {
            echo json_encode(['status' => false, 'msg' => 'No attendance records found for the selected date']);
            return;
        }
        
        // Delete student attendance records
        $this->db->where('date', $date);
        $student_result = $this->db->delete('student_attendences');
        $student_deleted = $this->db->affected_rows();
        
        // Delete staff attendance records
        $this->db->where('date', $date);
        $staff_result = $this->db->delete('staff_attendance');
        $staff_deleted = $this->db->affected_rows();
        
        $total_deleted = $student_deleted + $staff_deleted;
        
        // Log the deletion results
        error_log("Attendance Controller: Deleted {$student_deleted} student records and {$staff_deleted} staff records for date: {$date}");
        
        if ($total_deleted > 0) {
            echo json_encode(['status' => true, 'msg' => "Successfully deleted {$total_deleted} attendance records ({$student_deleted} students, {$staff_deleted} staff) for {$date}"]);
        } else {
            echo json_encode(['status' => false, 'msg' => 'Failed to delete attendance records']);
        }
    }
    
    /**
     * Mark checkout time for students and staff
     */
    public function markOutTime()
    {
        $id = $this->input->post('id');
        $type = $this->input->post('type');
        $date = date('Y-m-d');
        $current_time = date('Y-m-d H:i:s');
        
        if (empty($id) || empty($type)) {
            echo json_encode(['status' => false, 'msg' => 'Invalid input data']);
            return;
        }
        
        if ($type == 'student') {
            // Get student details
            $this->db->select('student_session.id as student_session_id, students.id, students.admission_no, students.firstname, students.middlename, students.lastname');
            $this->db->from('students');
            $this->db->join('student_session', 'student_session.student_id = students.id', 'left');
            $this->db->where('students.id', $id);
            $student = $this->db->get()->row();
            
            if (!$student) {
                echo json_encode(['status' => false, 'msg' => 'Student not found']);
                return;
            }
            
            // Check if attendance exists for today
            $this->db->where('student_session_id', $student->student_session_id);
            $this->db->where('date', $date);
            $attendance = $this->db->get('student_attendences')->row();
            
            if (!$attendance) {
                echo json_encode(['status' => false, 'msg' => 'No check-in record found for today. Please check-in first.']);
                return;
            }
            
            if (!empty($attendance->out_time)) {
                echo json_encode(['status' => false, 'msg' => 'Checkout already recorded for today at ' . date('h:i A', strtotime($attendance->out_time))]);
                return;
            }
            
            // Update checkout time
            $this->db->where('id', $attendance->id);
            $this->db->update('student_attendences', ['out_time' => $current_time]);
            
            if ($this->db->affected_rows() > 0) {
                $student_name = trim($student->firstname . ' ' . ($student->middlename ?? '') . ' ' . ($student->lastname ?? ''));
                echo json_encode([
                    'status' => true, 
                    'msg' => 'Checkout recorded successfully', 
                    'name' => $student_name,
                    'checkout_time' => date('h:i A', strtotime($current_time))
                ]);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Failed to record checkout']);
            }
            
        } else if ($type == 'staff') {
            // Get staff details
            $this->db->select('staff.id, staff.employee_id, staff.name, staff.surname');
            $this->db->where('staff.id', $id);
            $staff = $this->db->get('staff')->row();
            
            if (!$staff) {
                echo json_encode(['status' => false, 'msg' => 'Staff not found']);
                return;
            }
            
            // Check if attendance exists for today
            $this->db->where('staff_id', $staff->id);
            $this->db->where('date', $date);
            $attendance = $this->db->get('staff_attendance')->row();
            
            if (!$attendance) {
                echo json_encode(['status' => false, 'msg' => 'No check-in record found for today. Please check-in first.']);
                return;
            }
            
            if (!empty($attendance->out_time)) {
                echo json_encode(['status' => false, 'msg' => 'Checkout already recorded for today at ' . date('h:i A', strtotime($attendance->out_time))]);
                return;
            }
            
            // Update checkout time
            $this->db->where('id', $attendance->id);
            $this->db->update('staff_attendance', ['out_time' => $current_time]);
            
            if ($this->db->affected_rows() > 0) {
                $staff_name = trim($staff->name . ' ' . ($staff->surname ?? ''));
                echo json_encode([
                    'status' => true, 
                    'msg' => 'Checkout recorded successfully', 
                    'name' => $staff_name,
                    'checkout_time' => date('h:i A', strtotime($current_time))
                ]);
            } else {
                echo json_encode(['status' => false, 'msg' => 'Failed to record checkout']);
            }
        } else {
            echo json_encode(['status' => false, 'msg' => 'Invalid user type']);
        }
    }
    
    private function validateCheckinTime()
    {
        // Get camera settings for time validation
        $this->db->where('id', 1);
        $settings = $this->db->get('camera_settings')->row();
        
        if (!$settings) {
            // If no settings found, allow check-in (default behavior)
            return ['allowed' => true, 'message' => ''];
        }
        
        $current_time = date('H:i');
        $checkin_start = date('H:i', strtotime($settings->checkin_start_time));
        $checkin_end = date('H:i', strtotime($settings->checkin_end_time));
        
        if ($current_time < $checkin_start) {
            return [
                'allowed' => false, 
                'message' => "Check-in time has not started yet. Please check-in between {$checkin_start} and {$checkin_end}"
            ];
        } elseif ($current_time > $checkin_end) {
            return [
                'allowed' => false, 
                'message' => "Check-in time has ended. Check-in is allowed between {$checkin_start} and {$checkin_end}"
            ];
        }
        
        return ['allowed' => true, 'message' => ''];
    }
    
}
