<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Feereminder extends Admin_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->load->model('feereminder_model');
        $this->load->model('studentfeemaster_model');
        $this->load->library('smsgateway');
        $this->load->library('mailer');
    }

    public function setting()
    {
        if (!$this->rbac->hasPrivilege('fees_reminder', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'Fees Collection');
        $this->session->set_userdata('sub_menu', 'feereminder/setting');
        $data = array();

        $feereminderlist         = $this->feereminder_model->get();
        $data['feereminderlist'] = $feereminderlist;

        if ($this->input->server('REQUEST_METHOD') == "POST") {

            $ids          = $this->input->post('ids');
            $update_array = array();
            foreach ($ids as $id_key => $id_value) {
                $array = array(
                    'id'        => $id_value,
                    'is_active' => 0,
                    'day'       => $this->input->post('days' . $id_value),
                );
                $is_active = $this->input->post('isactive_' . $id_value);

                if (isset($is_active)) {
                    $array['is_active'] = $is_active;
                }

                $update_array[] = $array;
            }

            $this->feereminder_model->updatebatch($update_array);
            $this->session->set_flashdata('msg', '<div class="alert alert-success">' . $this->lang->line('update_message') . '</div>');
            redirect('admin/feereminder/setting');
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/feereminder/setting', $data);
        $this->load->view('layout/footer', $data);
    }

    public function get_pending_count()
    {
        header('Content-Type: application/json');
        
        if (!$this->rbac->hasPrivilege('fees_reminder', 'can_view')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            $count = $this->studentfeemaster_model->getPendingFeesCount();
            echo json_encode([
                'status' => 'success',
                'count' => (int)$count
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error getting pending fees count: ' . $e->getMessage()
            ]);
        }
    }

    public function send_immediate()
    {
        // Prevent any output buffering issues
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        header('Content-Type: application/json');
        header('Cache-Control: no-cache, must-revalidate');
        
        if (!$this->rbac->hasPrivilege('fees_reminder', 'can_edit')) {
            $response = ['status' => 'error', 'message' => 'Access denied'];
            echo json_encode($response);
            die();
        }

        if ($this->input->server('REQUEST_METHOD') != "POST") {
            $response = ['status' => 'error', 'message' => 'Invalid request method'];
            echo json_encode($response);
            die();
        }

        try {
            $custom_message = $this->input->post('custom_message');
            
            // Get students with pending fees (limited for performance)
            $pending_students = $this->studentfeemaster_model->getPendingFees();
            
            if (empty($pending_students)) {
                $response = [
                    'status' => 'success',
                    'message' => 'No students with pending fees found',
                    'sent_count' => 0
                ];
                echo json_encode($response);
                die();
            }

            $sent_count = 0;
            $sms_count = 0;
            $email_count = 0;
            $total_students = count($pending_students);
            
            $default_message = "Dear Parent, This is a reminder that your ward {student_name}'s fee payment of Rs.{pending_amount} is pending for class {class}. Please pay at your earliest convenience. Thank you.";
            $message = !empty($custom_message) ? $custom_message : $default_message;

            // Process all students with pending fees
            $students_to_process = $pending_students;

            foreach ($students_to_process as $student) {
                // Prepare message with student details
                $personalized_message = str_replace(
                    ['{student_name}', '{class}', '{pending_amount}'],
                    [$student->firstname . ' ' . $student->lastname, $student->class . '-' . $student->section, $student->pending_amount],
                    $message
                );

                // Send SMS if mobile number exists
                if (!empty($student->mobileno)) {
                    try {
                        $this->smsgateway->sendSMS($student->mobileno, $personalized_message, '', '');
                        $sms_count++;
                        $sent_count++;
                    } catch (Exception $sms_error) {
                        log_message('error', 'SMS sending failed for ' . $student->mobileno . ': ' . $sms_error->getMessage());
                    }
                }

                // Send Email if email exists  
                if (!empty($student->email)) {
                    try {
                        $subject = "Fee Payment Reminder - " . $student->firstname . ' ' . $student->lastname;
                        $this->mailer->send_mail($student->email, $subject, $personalized_message);
                        $email_count++;
                    } catch (Exception $email_error) {
                        log_message('error', 'Email sending failed for ' . $student->email . ': ' . $email_error->getMessage());
                    }
                }
            }

            // Create detailed Hindi message
            $hindi_message = "कुल छात्र: {$total_students} | SMS भेजे गए: {$sms_count} | Email भेजे गए: {$email_count} | कुल संदेश: {$sent_count}";
            
            $response = [
                'status' => 'success',
                'message' => $hindi_message,
                'sent_count' => $sent_count,
                'total_students' => $total_students,
                'sms_count' => $sms_count,
                'email_count' => $email_count,
                'details' => "Total {$total_students} students found with pending fees. SMS sent to {$sms_count} students, Email sent to {$email_count} students."
            ];
            echo json_encode($response);
            die();

        } catch (Exception $e) {
            $response = [
                'status' => 'error',
                'message' => 'Error sending reminders: ' . $e->getMessage()
            ];
            echo json_encode($response);
            die();
        }
    }

    public function test_real_pending_calculation()
    {
        // Test method to verify real pending amount calculations
        if (!$this->rbac->hasPrivilege('fees_reminder', 'can_view')) {
            access_denied();
        }

        echo "<h2>Testing Real Pending Amount Calculations</h2>";
        
        try {
            // Test pending fees count
            echo "<h3>1. Testing getPendingFeesCount()</h3>";
            $count = $this->studentfeemaster_model->getPendingFeesCount();
            echo "<p><strong>Pending Fees Count:</strong> {$count} students</p>";
            
            // Test pending fees details
            echo "<h3>2. Testing getPendingFees() - First 10 Records</h3>";
            $pending_students = $this->studentfeemaster_model->getPendingFees();
            
            if (!empty($pending_students)) {
                echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
                echo "<tr style='background-color: #f2f2f2;'>";
                echo "<th>Student ID</th><th>Name</th><th>Class</th><th>Mobile</th><th>Pending Amount</th>";
                echo "</tr>";
                
                $displayed = 0;
                foreach ($pending_students as $student) {
                    if ($displayed >= 10) break; // Show only first 10
                    
                    echo "<tr>";
                    echo "<td>{$student->student_id}</td>";
                    echo "<td>{$student->firstname} {$student->lastname}</td>";
                    echo "<td>{$student->class}-{$student->section}</td>";
                    echo "<td>{$student->mobileno}</td>";
                    echo "<td>₹" . number_format($student->pending_amount, 2) . "</td>";
                    echo "</tr>";
                    $displayed++;
                }
                echo "</table>";
                
                // Calculate total pending amount
                $total_pending = 0;
                foreach ($pending_students as $student) {
                    $total_pending += $student->pending_amount;
                }
                
                echo "<p><strong>Total Students with Pending Fees:</strong> " . count($pending_students) . "</p>";
                echo "<p><strong>Total Pending Amount:</strong> ₹" . number_format($total_pending, 2) . "</p>";
                
            } else {
                echo "<p style='color: green;'><strong>✓ No students have pending fees!</strong></p>";
            }
            
            // Test simple fallback method
            echo "<h3>3. Testing getPendingFeesSimple() Fallback</h3>";
            $simple_students = $this->studentfeemaster_model->getPendingFeesSimple();
            echo "<p><strong>Simple Method Count:</strong> " . count($simple_students) . " students</p>";
            
            // Test count simple fallback method
            echo "<h3>4. Testing getPendingFeesCountSimple() Fallback</h3>";
            $simple_count = $this->studentfeemaster_model->getPendingFeesCountSimple();
            echo "<p><strong>Simple Count Method:</strong> {$simple_count} students</p>";
            
            echo "<hr>";
            echo "<h3>Summary</h3>";
            echo "<ul>";
            echo "<li><strong>Advanced Method:</strong> {$count} students with pending fees</li>";
            echo "<li><strong>Simple Method:</strong> {$simple_count} students with pending fees</li>";
            echo "<li><strong>Total Pending Amount:</strong> ₹" . number_format($total_pending, 2) . "</li>";
            echo "</ul>";
            
            echo "<p style='color: blue;'><strong>✓ Real pending amount calculation is now working!</strong></p>";
            echo "<p><em>The system now calculates actual pending amounts instead of using hardcoded ₹1000.</em></p>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'><strong>Error:</strong> " . $e->getMessage() . "</p>";
        }
        
        echo "<br><a href='" . site_url('admin/feereminder/setting') . "' class='btn btn-primary'>← Back to Fee Reminder Settings</a>";
    }

}
